package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Strings;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.Threshold;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ApiModel(value = "ThresholdAlertTemplate")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ThresholdAlertTemplateDto {
    @ApiModelProperty(
            value = "Label selectors to define metrics to check",
            required = true)
    public String selectors;

    @ApiModelProperty(
            value = "Transformations that are applied before testing predicates rules")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public String transformations;

    @ApiModelProperty(
            value = "A list of predicate rules to test against the data")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TemplatePredicateRuleDto> predicateRules;

    public static ThresholdAlertTemplateDto fromProto(@Nonnull Threshold proto) {
        ThresholdAlertTemplateDto dto = new ThresholdAlertTemplateDto();
        dto.selectors = proto.getSelectors();
        dto.transformations = Strings.emptyToNull(proto.getTransformations());
        dto.predicateRules = proto.getPredicateRulesCount() == 0 ? null : proto.getPredicateRulesList().stream()
                .map(TemplatePredicateRuleDto::fromProto)
                .collect(Collectors.toList());

        return dto;
    }

    public Threshold toProto() {
        boolean predicateRulesDefined = predicateRules != null && predicateRules.size() > 0;
        if (!predicateRulesDefined) {
            throw new IllegalArgumentException("At least one predicate rule should be specified");
        }
        return Threshold.newBuilder()
                .setSelectors(Nullables.orEmpty(selectors))
                .setTransformations(Nullables.orEmpty(transformations))
                .addAllPredicateRules(Nullables.orEmpty(predicateRules).stream().map(TemplatePredicateRuleDto::toProto).collect(Collectors.toList()))
                .build();
    }
}
