package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Strings;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel(value = "Threshold")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ThresholdDto {
    // TODO: maybe better use list dto where define how match each label (gordiychuk@)
    @ApiModelProperty(
            value = "Label selectors to define metrics to check",
            required = true,
            position = 0)
    public String selectors;

    @ApiModelProperty(
            value = "Define criteria by that will be check alert",
            position = 2)
    public EThresholdType timeAggregation;

    @ApiModelProperty(
            value = "Predicate uses for compare point with threshold",
            position = 3)
    public ECompare predicate;

    @ApiModelProperty(
            value = "Target threshold that will be use with metrics data on specified period",
            position = 4)
    public Double threshold;

    @ApiModelProperty(
            value = "Transformations that are applied before testing predicates rules",
            position = 5)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public String transformations;

    @ApiModelProperty(
            value = "A list of predicate rules to test against the data",
            position = 6)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<PredicateRuleDto> predicateRules;

    public static ThresholdDto fromProto(@Nonnull TThreshold proto) {
        ThresholdDto dto = new ThresholdDto();
        dto.selectors = Selectors.format(LabelSelectorConverter.protoToSelectors(proto.getNewSelectors()));
        dto.timeAggregation = proto.getThresholdType();
        dto.predicate = proto.getComparison();
        dto.threshold = proto.getThreshold();
        dto.transformations = Strings.emptyToNull(proto.getTransformations());
        dto.predicateRules = proto.getPredicateRulesCount() == 0 ? null : proto.getPredicateRulesList().stream()
                        .map(PredicateRuleDto::fromProto)
                        .collect(Collectors.toList());

        return dto;
    }

    public TThreshold toProto() {
        boolean predicateRulesDefined = predicateRules != null && predicateRules.size() > 0;
        boolean simplePredicateDefined = (predicate != null) && (threshold != null) && (timeAggregation != null);
        if (!predicateRulesDefined && !simplePredicateDefined) {
            throw new IllegalArgumentException("At least one predicate rule should be specified");
        }

        Selectors parsedSelectors = Selectors.parse(Nullables.orEmpty(selectors));
        if (parsedSelectors.isEmpty()) {
            throw new IllegalArgumentException("Selectors should be specified for threshold alert");
        }
        TThreshold.Builder builder = TThreshold.newBuilder()
                .setNewSelectors(LabelSelectorConverter.selectorsToNewProto(parsedSelectors))
                .setTransformations(Nullables.orEmpty(transformations))
                .addAllPredicateRules(Nullables.orEmpty(predicateRules).stream().map(PredicateRuleDto::toProto).collect(Collectors.toList()));

        if (timeAggregation != null) {
            builder.setThresholdType(timeAggregation);
        }
        if (predicate != null) {
            builder.setComparison(predicate);
        }
        if (threshold != null) {
            builder.setThreshold(threshold);
        }

        return builder.build();
    }
}
