package ru.yandex.solomon.alert.gateway.dto.alert.state;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.AlertMuteStatus;
import ru.yandex.solomon.alert.protobuf.TEvaluationState;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("AlertEvaluationState")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertEvaluationStateDto {
    @ApiModelProperty(
            value = "Unique identity for alert",
            dataType = "java.util.UUID",
            position = 1)
    public String alertId;

    @ApiModelProperty(
            value = "Reference to project that contains alert",
            readOnly = true,
            required = true,
            dataType = "java.util.UUID",
            position = 2)
    public String projectId;

    @ApiModelProperty(
            value = "Version of alert that will be use to evaluate latest time." +
                    "Each change of alert reset state.",
            position = 3)
    public int alertVersion;

    @ApiModelProperty(
            value = "Evaluation status and details. It's status of latest change between statuses, " +
                    "for example from OK to ALARM",
            position = 4)
    public AlertEvaluationStatusDto status;

    @ApiModelProperty(
            value = "Time when evaluation state will changed latest time, for example from OK to ALARM",
            dataType = "java.time.Instant",
            position = 5)
    public String since;

    @ApiModelProperty(
            value = "UNIX time when latest alert evaluation ocurrs." +
                    "Since - LatestEval = how much time alert in particular state",
            dataType = "java.time.Instant",
            position = 6)
    public String latestEval;

    @ApiModelProperty(
            value = "Evaluation status and details. It's status of previous change between statuses, for example " +
                    "from OK to ALARM",
            position = 7)
    public AlertEvaluationStatusDto previousStatus;

    @ApiModelProperty(
            value = "Mute status",
            position = 8)
    public AlertMuteStatusDto muteStatus;

    public static AlertEvaluationStateDto fromProto(@Nonnull TEvaluationState proto, @Nonnull AlertMuteStatus mute) {
        AlertEvaluationStateDto dto = new AlertEvaluationStateDto();
        dto.alertId = proto.getAlertId();
        dto.projectId = proto.getProjectId();
        dto.alertVersion = proto.getAlertVersion();
        dto.status = AlertEvaluationStatusDto.fromProto(proto.getStatus());
        dto.since = Instant.ofEpochMilli(proto.getSinceMillis()).toString();
        dto.latestEval = Instant.ofEpochMilli(proto.getLatestEvalMillis()).toString();
        dto.previousStatus = AlertEvaluationStatusDto.fromProto(proto.getPreviousStatus());
        dto.muteStatus = AlertMuteStatusDto.fromProto(mute);
        return dto;
    }
}
