package ru.yandex.solomon.alert.gateway.dto.alert.state;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.TNotificationState;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("AlertNotificationState")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertNotificationStateDto {
    public Map<String, NotificationChannelStateDto> stateByChannelId;

    public static AlertNotificationStateDto fromProto(List<TNotificationState> statesList) {
        AlertNotificationStateDto dto = new AlertNotificationStateDto();
        dto.stateByChannelId = statesList.stream()
                .collect(Collectors.toMap(
                        TNotificationState::getNotificationChannelId,
                        NotificationChannelStateDto::fromProto));
        return dto;
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class NotificationChannelStateDto {
        @ApiModelProperty(
                value = "Reference to project that contains alert",
                readOnly = true,
                required = true,
                dataType = "java.util.UUID",
                position = 1)
        public String projectId;

        @ApiModelProperty(
                value = "Reference to notification channel",
                readOnly = true,
                required = true,
                dataType = "java.util.UUID",
                position = 2)
        public String channelId;

        @ApiModelProperty(
                value = "Latest evaluation status",
                readOnly = true,
                required = true,
                dataType = "java.util.UUID",
                position = 3)
        public NotificationStatusDto status;

        @ApiModelProperty(
                value = "Time when notification was triggered latest time",
                dataType = "java.time.Instant",
                position = 4)
        public String latestEval;

        @ApiModelProperty(
                value = "Time when notification was successfully delivered latest time",
                dataType = "java.time.Instant",
                position = 5)
        public String latestSuccess;

        public static NotificationChannelStateDto fromProto(@Nonnull TNotificationState proto) {
            NotificationChannelStateDto dto = new NotificationChannelStateDto();
            dto.projectId = proto.getProjectId();
            dto.channelId = proto.getNotificationChannelId();
            dto.status = NotificationStatusDto.fromProto(proto.getStatus());
            dto.latestEval = Instant.ofEpochMilli(proto.getLatestEvalMillis()).toString();
            dto.latestSuccess = Instant.ofEpochMilli(proto.getLatestSuccessMillis()).toString();
            return dto;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class NotificationStatusDto {
        public TNotificationStatus.ECode code;
        public String description;

        public static NotificationStatusDto fromProto(@Nonnull TNotificationStatus proto) {
            NotificationStatusDto dto = new NotificationStatusDto();
            dto.code = proto.getCode();
            dto.description = proto.getDesctiption();
            return dto;
        }
    }
}
