package ru.yandex.solomon.alert.gateway.dto.mute;

import java.time.Instant;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.MuteStatus;
import ru.yandex.solomon.alert.protobuf.mute.Mute;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("Mute")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class MuteDto {
    @ApiModelProperty(
            value = "Mute unique identifier",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            required = true)
    public String id;

    @ApiModelProperty(
            value = "The project that this mute belongs to",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            required = true,
            position = 1)
    public String projectId;

    @ApiModelProperty(
            value = "Name of this mute",
            position = 6)
    public String name;

    @ApiModelProperty(
            value = "Description of this mute",
            position = 2)
    public String description;

    @ApiModelProperty(
            value = "Related Startrek ticket",
            position = 3)
    public String ticketId;

    @ApiModelProperty(
            value = "Mute starting time",
            position = 4)
    public Instant from;

    @ApiModelProperty(
            value = "Mute finishing time",
            position = 5)
    public Instant to;

    @ApiModelProperty(
            value = "Type specific configuration",
            required = true,
            position = 10)
    public Type type;

    @ApiModelProperty(
            value = "Mute status at the time of the request",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            position = 20)
    public MuteStatus status;

    @ApiModelProperty(
            value = "Version of this mute",
            position = 50)
    public Integer version;

    @ApiModelProperty(
            value = "User who created this mute",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            position = 51)
    public String createdBy;

    @ApiModelProperty(
            value = "Mute creation time",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            position = 52)
    public Instant createdAt;

    @ApiModelProperty(
            value = "User who updated this mute for the last time",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            position = 53)
    public String updatedBy;

    @ApiModelProperty(
            value = "Mute last update time",
            accessMode = ApiModelProperty.AccessMode.READ_ONLY,
            position = 54)
    public Instant updatedAt;

    public static MuteDto fromProto(@Nonnull Mute proto) {
        var dto = new MuteDto();

        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.name = proto.getName();
        dto.description = proto.getDescription();
        dto.ticketId = proto.getTicketId();
        dto.from = Instant.ofEpochMilli(proto.getFromMillis());
        dto.to = Instant.ofEpochMilli(proto.getToMillis());

        dto.status = proto.getStatus();

        dto.version = proto.getVersion();
        dto.createdBy = proto.getCreatedBy();
        dto.createdAt = Instant.ofEpochMilli(proto.getCreatedAt());
        dto.updatedBy = proto.getUpdatedBy();
        dto.updatedAt = Instant.ofEpochMilli(proto.getUpdatedAt());

        dto.type = Type.fromProto(proto);

        return dto;
    }

    public Mute.Builder toProtoBuilder() {
        var builder = Mute.newBuilder()
                .setId(Nullables.orEmpty(id))
                .setProjectId(Nullables.orEmpty(projectId))
                .setName(Nullables.orEmpty(name))
                .setDescription(Nullables.orEmpty(description))
                .setTicketId(Nullables.orEmpty(ticketId))
                .setFromMillis(Optional.ofNullable(from).map(Instant::toEpochMilli).orElse(0L))
                .setToMillis(Optional.ofNullable(to).map(Instant::toEpochMilli).orElse(0L))
                .setVersion(Nullables.orZero(version))
                .setCreatedBy(Nullables.orEmpty(createdBy))
                .setCreatedAt(Optional.ofNullable(createdAt).map(Instant::toEpochMilli).orElse(0L))
                .setUpdatedBy(Nullables.orEmpty(updatedBy))
                .setUpdatedAt(Optional.ofNullable(updatedAt).map(Instant::toEpochMilli).orElse(0L));

        if (type != null) {
            type.fillProto(builder);
        }

        return builder;
    }

    @ApiModel(value = "MuteType", description = "Type specific mute configuration")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Type {
        @ApiModelProperty(
                value = "Compare values of a metrics with a user defined threshold",
                position = 1)
        public SelectorsMuteDto selectors;

        public static Type fromProto(@Nonnull Mute mute) {
            var type = new Type();
            switch (mute.getTypeCase()) {
                case SELECTORS -> type.selectors = SelectorsMuteDto.fromProto(mute.getSelectors());
                default -> throw new UnsupportedOperationException("Unsupported mute type: " + mute);
            }
            return type;
        }

        public void fillProto(@Nonnull Mute.Builder proto) {
            if (selectors != null) {
                proto.setSelectors(selectors.toProtoBuilder());
            }
        }
    }
}
