package ru.yandex.solomon.alert.gateway.dto.mute;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.mute.SelectorsType;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsException;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("SelectorsMute")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class SelectorsMuteDto {
    @ApiModelProperty(
            value = "Alert selector, may match any number of alerts. Key must be 'alert'",
            example = "alert=uptime_alert")
    public String alertSelector;

    @ApiModelProperty(
            value = "Selectors to match subalert labels, may be empty",
            example = "host=Sas, service=coremon",
            position = 1)
    public String labelSelectors;

    public static SelectorsMuteDto fromProto(@Nonnull SelectorsType proto) {
        var dto = new SelectorsMuteDto();
        dto.alertSelector = SelectorsFormat.format(LabelSelectorConverter.protoToSelector(proto.getAlertSelector()));
        dto.labelSelectors = SelectorsFormat.format(LabelSelectorConverter.protoToSelectors(proto.getLabelSelectors()));
        return dto;
    }

    private static Selector parseAlertSelector(@Nonnull String alertSelector) {
        try {
            var selector = SelectorsFormat.parseSelector(alertSelector);
            if (!selector.getKey().equals("alert")) {
                throw new BadRequestException("Alert selector should have key `alert`, got `" + selector.getKey() + "`");
            }
            return selector;
        } catch (SelectorsException e) {
            throw new BadRequestException("Alert selector should be a valid selector with key `alert`. Got exception: " +
                    e.getMessage());
        }
    }

    public SelectorsType.Builder toProtoBuilder() {
        Selector alertSelector = Optional.ofNullable(this.alertSelector)
                .map(SelectorsMuteDto::parseAlertSelector)
                .orElse(Selector.any("alert"));

        Selectors labelSelectors = SelectorsFormat.parse(Nullables.orEmpty(this.labelSelectors));

        return SelectorsType.newBuilder()
                .setAlertSelector(LabelSelectorConverter.selectorToProto(alertSelector))
                .setLabelSelectors(LabelSelectorConverter.selectorsToNewProto(labelSelectors));
    }
}
