package ru.yandex.solomon.alert.gateway.dto.notificationChannel;

import java.time.Instant;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.collect.ImmutableList;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("NotificationChannel")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class NotificationChannelDto {
    @ApiModelProperty(
            value = "unique identity for notification",
            readOnly = true,
            required = true,
            example = "d7d55c93-66aa-434d-b56c-f8b28c26cce4",
            position = 0)
    public String id;

    @ApiModelProperty(
            value = "ProjectId own this notification",
            readOnly = true,
            required = true,
            example = "solomon",
            position = 1)
    public String projectId;

    @ApiModelProperty(
            value = "Human-readable name of notification",
            required = true,
            example = "@telegram-oops",
            position = 2)
    public String name;

    @ApiModelProperty(
            value = "actual version of notification",
            example = "42",
            position = 3)
    public Integer version;

    @ApiModelProperty(
            value = "TNotification will send event only about specified alert evaluation status",
            position = 4)
    public List<TEvaluationStatus.ECode> notifyAboutStatuses;

    @ApiModelProperty(
            value = "Delay between repeated notify about the same evaluation status, by default notify about" +
                    "particular evaluation status only after change it. Negative duration means that repeat" +
                    "notification not used.",
            example = "600000",
            position = 7)
    public Long repeatNotifyDelayMillis;

    @ApiModelProperty(
            value = "User created notification channel",
            readOnly = true,
            position = 8)
    public String createdBy;

    @ApiModelProperty(
            value = "Time when notification was created",
            readOnly = true,
            dataType = "date-time",
            example = "2017-12-19T12:50:53.583Z",
            position = 8)
    public String createdAt;

    @ApiModelProperty(
            value = "User modified notification channel",
            readOnly = true,
            position = 9)
    public String updatedBy;

    @ApiModelProperty(
            value = "Time when notification was updated last time",
            readOnly = true,
            dataType = "ISO-8601",
            example = "2017-12-19T12:55:53.583Z",
            position = 9)
    public String updatedAt;

    @ApiModelProperty(
            value = "Default channel for project",
            position = 10)
    public boolean isDefault;

    @ApiModelProperty(
            value = "Channel is default for alert severity.",
            position = 11)
    public List<Severity> defaultForAlertSeverity;


    @ApiModelProperty(value = "Channel specific configuration", required = true, position = 8)
    public NotificationMethod method;

    public static NotificationChannelDto fromProto(@Nonnull TNotification proto) {
        NotificationChannelDto dto = new NotificationChannelDto();
        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.name = proto.getName();
        dto.version = proto.getVersion();
        dto.notifyAboutStatuses = ImmutableList.copyOf(proto.getNotifyAboutStatusesList());
        dto.repeatNotifyDelayMillis = proto.getRepeatNotifyDelayMillis();
        dto.createdBy = proto.getCreatedBy();
        dto.createdAt = Instant.ofEpochMilli(proto.getCreatedAt()).toString();
        dto.updatedBy = proto.getUpdatedBy();
        dto.updatedAt = Instant.ofEpochMilli(proto.getUpdatedAt()).toString();
        dto.method = NotificationMethod.fromProto(proto);
        dto.isDefault = proto.getDefaultForProject();
        dto.defaultForAlertSeverity = proto.getDefaultForAlertSeverityList();
        return dto;
    }

    public TNotification toProto() {
        TNotification.Builder proto = TNotification.newBuilder();
        proto.setId(Nullables.orEmpty(id));
        proto.setProjectId(projectId);
        proto.setName(Nullables.orEmpty(name));
        proto.setVersion(Nullables.orDefault(version, 0));
        proto.addAllNotifyAboutStatuses(Nullables.orEmpty(notifyAboutStatuses));
        proto.setRepeatNotifyDelayMillis(Nullables.orDefault(repeatNotifyDelayMillis, -1));
        proto.setCreatedBy(Nullables.orEmpty(createdBy));
        proto.setUpdatedBy(Nullables.orEmpty(updatedBy));
        proto.setDefaultForProject(!Nullables.orEmpty(defaultForAlertSeverity).isEmpty() || isDefault);
        proto.addAllDefaultForAlertSeverity(Nullables.orEmpty(defaultForAlertSeverity));
        if (createdAt != null) {
            proto.setCreatedAt(Instant.parse(createdAt).toEpochMilli());
        }
        if (updatedAt != null) {
            proto.setUpdatedAt(Instant.parse(updatedAt).toEpochMilli());
        }

        if (method != null) {
            method.fillProto(proto);
        }

        return proto.build();
    }
}
