package ru.yandex.solomon.alert.gateway.endpoint;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertDto;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertExplainEvaluationDto;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertInterpolatedResponse;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertList;
import ru.yandex.solomon.alert.gateway.dto.alert.ProjectStatsDto;
import ru.yandex.solomon.alert.gateway.dto.alert.SubAlertDto;
import ru.yandex.solomon.alert.gateway.dto.alert.SubAlertList;
import ru.yandex.solomon.alert.gateway.dto.alert.UpdateAlertTemplateVersionDto;
import ru.yandex.solomon.alert.gateway.dto.alert.state.AlertEvaluationStateDto;
import ru.yandex.solomon.alert.gateway.dto.alert.state.AlertEvaluationStatsDto;
import ru.yandex.solomon.alert.gateway.dto.alert.state.AlertNotificationStateDto;
import ru.yandex.solomon.alert.gateway.dto.alert.state.AlertNotificationStatsDto;
import ru.yandex.solomon.alert.gateway.dto.mute.MuteDto;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.EOrderDirection;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.model.protobuf.Selector;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/alerts", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class AlertController {
    private static final Logger logger = LoggerFactory.getLogger(AlertController.class);
    private static final String SERVICE_PROVIDER_HEADER = "X-Service-Provider";
    private static final String IDEMPOTENCY_KEY_HEADER = "Idempotency-Key";

    private final ProjectsDao projectsDao;
    private final AlertApi api;
    private final Authorizer authorizer;

    @Autowired
    public AlertController(AlertApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        this.api = api;
        this.projectsDao = projectsDao;
        this.authorizer = authorizer;
    }

    @ApiOperation(value = "Get alert by id", response = AlertDto.class)
    @RequestMapping(path = "/{alertId}", method = RequestMethod.GET)
    CompletableFuture<AlertDto> getAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadAlertRequest request = TReadAlertRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readAlert(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertDto.fromProto(response.getAlert());
            });
    }

    @ApiOperation(value = "Get alert by id and it interpolated version", response = AlertInterpolatedResponse.class)
    @RequestMapping(path = "/{alertId}/interpolated", method = RequestMethod.GET)
    CompletableFuture<AlertInterpolatedResponse> getAlertInterpolated(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadAlertInterpolatedRequest request = TReadAlertInterpolatedRequest.newBuilder()
                .setAlertId(alertId)
                .setProjectId(projectId)
                .setServiceProvider(serviceProviderId)
                .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
                .thenCompose(aVoid -> api.readAlert(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertInterpolatedResponse.fromProto(response);
                });
    }

    @ApiOperation(value = "Get sub alert by id", response = AlertDto.class)
    @RequestMapping(path = "/{parentId}/subAlerts/{alertId}", method = RequestMethod.GET)
    CompletableFuture<SubAlertDto> getSubAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("parentId") String parentId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadSubAlertRequest request = TReadSubAlertRequest.newBuilder()
            .setProjectId(projectId)
            .setParentId(parentId)
            .setAlertId(alertId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readSubAlert(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return SubAlertDto.fromProto(response.getSubAlert());
            });
    }

    @ApiOperation(value = "Update alert by id", response = AlertDto.class)
    @RequestMapping(path = "/{alertId}", method = RequestMethod.PUT)
    public CompletableFuture<AlertDto> updateAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestBody AlertDto alert,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId,
            @RequestHeader(value = IDEMPOTENCY_KEY_HEADER, required = false, defaultValue = "") String idempotencyKey)
    {
        alert.id = alertId;
        alert.projectId = projectId;
        alert.updatedBy = subject.getUniqueId();
        alert.updatedAt = Instant.now().toString();

        TUpdateAlertRequest request = TUpdateAlertRequest.newBuilder()
                .setAlert(alert.toProto()
                        .toBuilder()
                        .setServiceProvider(serviceProviderId)
                        .build())
                .setIdempotentOperationId(idempotencyKey)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_UPDATE, serviceProviderId)
            .thenCompose(aVoid -> checkProjectExistence(projectId))
            .thenCompose(aVoid -> api.updateAlert(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertDto.fromProto(response.getAlert());
            });
    }

    @Deprecated
    @ApiOperation(value = "Update alert template version to specified", response = AlertDto.class)
    @RequestMapping(path = "/{alertId}/update-template-version", method = RequestMethod.POST)
    public CompletableFuture<AlertDto> updateTemplateVersionOld(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestBody UpdateAlertTemplateVersionDto dto,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        return updateTemplateVersionInner(subject, projectId, alertId, dto, serviceProviderId);
    }

    @ApiOperation(value = "Update alert template version to specified", response = AlertDto.class)
    @RequestMapping(path = "/{alertId}/updateTemplateVersion", method = RequestMethod.POST)
    public CompletableFuture<AlertDto> updateTemplateVersion(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestBody UpdateAlertTemplateVersionDto dto,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        return updateTemplateVersionInner(subject, projectId, alertId, dto, serviceProviderId);
    }

    private CompletableFuture<AlertDto> updateTemplateVersionInner(
            AuthSubject subject,
            String projectId,
            String alertId,
            UpdateAlertTemplateVersionDto dto,
            String serviceProviderId)
    {
        UpdateAlertTemplateVersionRequest request = UpdateAlertTemplateVersionRequest.newBuilder()
                .setTemplateId(Nullables.orEmpty(dto.templateId))
                .setTemplateVersionTag(Nullables.orEmpty(dto.newTemplateVersionTag))
                .setProjectId(projectId)
                .setAlertData(UpdateAlertTemplateVersionRequest.AlertData.newBuilder()
                        .setAlertId(alertId)
                        .setUpdatedBy(subject.getUniqueId())
                        .build())
                .setServiceProvider(serviceProviderId)
                .build();

        return authorize(subject, projectId, Permission.CONFIGS_UPDATE, serviceProviderId)
                .thenCompose(aVoid -> checkProjectExistence(projectId))
                .thenCompose(aVoid -> api.updateAlertTemplateVersion(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatusCode(), response::getStatusMessage);
                    return AlertDto.fromProto(response.getAlert());
                });
    }

    @RequestMapping(method = RequestMethod.POST)
    @ApiOperation(value = "Create new alert", response = AlertDto.class)
    CompletableFuture<AlertDto> createAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody AlertDto alert,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId,
            @RequestHeader(value = IDEMPOTENCY_KEY_HEADER, required = false, defaultValue = "") String idempotencyKey)
    {
        String now = Instant.now().toString();
        alert.projectId = projectId;
        alert.createdBy = subject.getUniqueId();
        alert.createdAt = now;
        alert.updatedBy = subject.getUniqueId();
        alert.updatedAt = now;

        TCreateAlertRequest request = TCreateAlertRequest.newBuilder()
                .setAlert(alert.toProto()
                        .toBuilder()
                        .setServiceProvider(serviceProviderId)
                        .build())
                .setIdempotentOperationId(idempotencyKey)
                .build();

        return authorize(subject, projectId, Permission.CONFIGS_CREATE, serviceProviderId)
            .thenCompose(aVoid -> checkProjectExistence(projectId))
            .thenCompose(aVoid -> api.createAlert(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertDto.fromProto(response.getAlert());
            });
    }

    @RequestMapping(path = "/{alertId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @ApiOperation(value = "Delete alert")
    CompletableFuture<Void> deleteAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId,
            @RequestHeader(value = IDEMPOTENCY_KEY_HEADER, required = false, defaultValue = "") String idempotencyKey)
    {
        TDeleteAlertRequest request = TDeleteAlertRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
                .setIdempotentOperationId(idempotencyKey)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_DELETE, serviceProviderId)
            .thenCompose(aVoid -> api.deleteAlert(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return null;
            });
    }

    // TODO: sub alert (gordiychuk@)
    @ApiOperation(value = "Get alert evaluation state by id", response = AlertEvaluationStateDto.class)
    @RequestMapping(path = "/{alertId}/state/evaluation", method = RequestMethod.GET)
    CompletableFuture<AlertEvaluationStateDto> getAlertEvaluationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadEvaluationStateRequest request = TReadEvaluationStateRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readEvaluationState(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertEvaluationStateDto.fromProto(response.getState(), response.getMuteStatus());
            });
    }

    @ApiOperation(value = "Get alert evaluation stats by id", response = AlertEvaluationStateDto.class)
    @RequestMapping(path = "/{alertId}/state/evaluationStats", method = RequestMethod.GET)
    CompletableFuture<AlertEvaluationStatsDto> getMultiAlertEvaluationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadEvaluationStatsRequest request = TReadEvaluationStatsRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readEvaluationStats(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertEvaluationStatsDto.fromProto(response);
            });
    }

    @ApiOperation(value = "Get sub alert evaluation state by id", response = AlertEvaluationStateDto.class)
    @RequestMapping(path = "/{parentId}/subAlerts/{alertId}/state/evaluation", method = RequestMethod.GET)
    CompletableFuture<AlertEvaluationStateDto> getSubAlertEvaluationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("parentId") String parentId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadEvaluationStateRequest request = TReadEvaluationStateRequest.newBuilder()
            .setProjectId(projectId)
            .setParentId(parentId)
            .setAlertId(alertId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readEvaluationState(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertEvaluationStateDto.fromProto(response.getState(), response.getMuteStatus());
            });
    }

    @ApiOperation(value = "Get alert notification state by id", response = AlertNotificationStateDto.class)
    @RequestMapping(path = "/{alertId}/state/notification", method = RequestMethod.GET)
    CompletableFuture<AlertNotificationStateDto> getAlertNotificationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadNotificationStateRequest request = TReadNotificationStateRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readNotificationState(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertNotificationStateDto.fromProto(response.getStatesList());
            });
    }

    @ApiOperation(value = "Get alert notification stats by id", response = AlertNotificationStateDto.class)
    @RequestMapping(path = "/{alertId}/state/notificationStats", method = RequestMethod.GET)
    CompletableFuture<AlertNotificationStatsDto> getAlertNotificationStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadNotificationStatsRequest request = TReadNotificationStatsRequest.newBuilder()
            .setAlertId(alertId)
            .setProjectId(projectId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readNotificationStats(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertNotificationStatsDto.fromProto(response);
            });
    }

    @ApiOperation(value = "Get sub alert notification state by id", response = AlertNotificationStateDto.class)
    @RequestMapping(path = "/{parentId}/subAlerts/{alertId}/state/notification", method = RequestMethod.GET)
    CompletableFuture<AlertNotificationStateDto> getSubAlertNotificationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("parentId") String parentId,
            @PathVariable("alertId") String alertId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        TReadNotificationStateRequest request = TReadNotificationStateRequest.newBuilder()
            .setProjectId(projectId)
            .setParentId(parentId)
            .setAlertId(alertId)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.readNotificationState(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertNotificationStateDto.fromProto(response.getStatesList());
            });
    }

    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "List alerts", response = AlertList.class)
    CompletableFuture<AlertList> listAlertsByProject(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByStates", required = false) List<EAlertState> filterByState,
            @RequestParam(value = "filterByTypes", required = false) List<EAlertType> filterByType,
            @RequestParam(value = "filterByEvaluationStatus", required = false) List<TEvaluationStatus.ECode> filterByEvaluation,
            @RequestParam(value = "filterByNotificationId", required = false) List<String> filterByNotificationId,
            @RequestParam(value = "filterByNotificationStatus", required = false) List<TNotificationStatus.ECode> filterByNotificationStatus,
            @RequestParam(value = "filterByMuteId", required = false) List<String> filterByMuteId,
            @RequestParam(value = "labelsSelector", required = false, defaultValue = "") String labelsSelector,
            @RequestParam(value = "orderByName", required = false) EOrderDirection orderByName,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken,
            @RequestParam(value = "templateServiceProviderId", defaultValue = "", required = false) String templateServiceProviderId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderHeader)
    {
        TListAlertRequest request = TListAlertRequest.newBuilder()
            .setProjectId(projectId)
            .setPageToken(pageToken)
            .setPageSize(pageSize)
            .setFilterByName(filterByName)
            .addAllFilterByState(Nullables.orEmpty(filterByState))
            .addAllFilterByType(Nullables.orEmpty(filterByType))
            .addAllFilterByEvaluationStatus(Nullables.orEmpty(filterByEvaluation))
            .addAllFilterByNotificationId(Nullables.orEmpty(filterByNotificationId))
            .addAllFilterByNotificationStatus(Nullables.orEmpty(filterByNotificationStatus))
            .setFilterByMuteReference(TListAlertRequest.MuteReference.newBuilder().addAllIds(Nullables.orEmpty(filterByMuteId)))
            .setOrderByName(Nullables.orDefault(orderByName, EOrderDirection.ASC))
            .setFilterCreatedByServiceProvider(serviceProviderHeader)
            .setTemplateProvider(templateServiceProviderId)
            .setLabelsSelector(labelsSelector)
            .setFullResultModel(false)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderHeader)
            .thenCompose(aVoid -> api.listAlerts(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertList.fromProto(response);
            });
    }

    @RequestMapping(path = "/explainMute", method = RequestMethod.POST)
    @ApiOperation(value = "List alerts that are affected by mute", response = AlertList.class)
    CompletableFuture<AlertList> explainMute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByStates", required = false) List<EAlertState> filterByState,
            @RequestParam(value = "filterByTypes", required = false) List<EAlertType> filterByType,
            @RequestParam(value = "filterByEvaluationStatus", required = false) List<TEvaluationStatus.ECode> filterByEvaluation,
            @RequestParam(value = "filterByNotificationId", required = false) List<String> filterByNotificationId,
            @RequestParam(value = "filterByNotificationStatus", required = false) List<TNotificationStatus.ECode> filterByNotificationStatus,
            @RequestParam(value = "orderByName", required = false) EOrderDirection orderByName,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken,
            @RequestParam(value = "templateServiceProviderId", defaultValue = "", required = false) String templateServiceProviderId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderHeader,
            @RequestBody MuteDto mute)
    {
        TListAlertRequest request = TListAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setPageToken(pageToken)
                .setPageSize(pageSize)
                .setFilterByName(filterByName)
                .addAllFilterByState(Nullables.orEmpty(filterByState))
                .addAllFilterByType(Nullables.orEmpty(filterByType))
                .addAllFilterByEvaluationStatus(Nullables.orEmpty(filterByEvaluation))
                .addAllFilterByNotificationId(Nullables.orEmpty(filterByNotificationId))
                .addAllFilterByNotificationStatus(Nullables.orEmpty(filterByNotificationStatus))
                .setFilterByInlinedMute(mute.toProtoBuilder())
                .setOrderByName(Nullables.orDefault(orderByName, EOrderDirection.ASC))
                .setFilterCreatedByServiceProvider(serviceProviderHeader)
                .setTemplateProvider(templateServiceProviderId)
                .setFullResultModel(false)
                .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderHeader)
                .thenCompose(aVoid -> api.listAlerts(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertList.fromProto(response);
                });
    }

    @RequestMapping(path = "/{parentId}/subAlerts", method = RequestMethod.GET)
    @ApiOperation(value = "List sub alerts", response = SubAlertList.class)
    CompletableFuture<SubAlertList> listAlertsByParent(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("parentId") String parentId,
            @RequestParam(value = "filterByEvaluationStatus", required = false) List<TEvaluationStatus.ECode> filterByEvaluation,
            @RequestParam(value = "filterByNotificationIds", required = false) List<String> filterByNotificationIds,
            @RequestParam(value = "filterByNotificationStatus", required = false) List<TNotificationStatus.ECode> filterByNotificationStatus,
            @RequestParam(value = "filterByLabels", required = false) String filterByLabels,
            @RequestParam(value = "orderByLabels", required = false) EOrderDirection orderByLabels,
            @RequestParam(value = "annotationKeys", required = false) List<String> annotationKeys,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        final List<Selector> protoSelectors;

        if (filterByLabels == null) {
            protoSelectors = Collections.emptyList();
        } else {
            Selectors selectors = Selectors.parse(Nullables.orEmpty(filterByLabels));
            protoSelectors = LabelSelectorConverter.selectorsToProto(selectors);
        }

        TListSubAlertRequest request = TListSubAlertRequest.newBuilder()
            .setProjectId(projectId)
            .setParentId(parentId)
            .addAllFilterByEvaluationStatus(Nullables.orEmpty(filterByEvaluation))
            .addAllFilterByLabels(protoSelectors)
            .addAllFilterByNotificationId(Nullables.orEmpty(filterByNotificationIds))
            .addAllFilterByNotificationStatus(Nullables.orEmpty(filterByNotificationStatus))
            .setOrderByLabels(Nullables.orDefault(orderByLabels, EOrderDirection.ASC))
            .addAllAnnotationKeys(Nullables.orEmpty(annotationKeys))
            .setPageToken(pageToken)
            .setPageSize(pageSize)
            .setServiceProvider(serviceProviderId)
            .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
            .thenCompose(aVoid -> api.listSubAlerts(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return SubAlertList.fromProto(response);
            });
    }

    @RequestMapping(path = "/{alertId}/explainEvaluation", method = RequestMethod.GET)
    @ApiOperation(value = "Explain evaluation for exist alert", response = AlertExplainEvaluationDto.class)
    CompletableFuture<AlertExplainEvaluationDto> explainEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "time", required = false, defaultValue = "") String time)
    {
        TExplainEvaluationRequest request = TExplainEvaluationRequest.newBuilder()
                .setReference(TExplainEvaluationRequest.TReferenceToAlert.newBuilder()
                        .setProjectId(projectId)
                        .setAlertId(alertId)
                        .build())
                .setEvaluationTimeMillis(time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli())
                .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L))
                .build();
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> explain(request));
    }

    @ApiOperation(value = "Explain evaluation for exist sub alert", response = AlertExplainEvaluationDto.class)
    @RequestMapping(path = "/{parentId}/subAlerts/{alertId}/explainEvaluation", method = RequestMethod.GET)
    CompletableFuture<AlertExplainEvaluationDto> explainEvaluationSubAlert(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("parentId") String parentId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "time", required = false, defaultValue = "") String time)
    {
        TExplainEvaluationRequest request = TExplainEvaluationRequest.newBuilder()
                .setReference(TExplainEvaluationRequest.TReferenceToAlert.newBuilder()
                        .setProjectId(projectId)
                        .setAlertId(alertId)
                        .setParentId(parentId)
                        .build())
                .setEvaluationTimeMillis(time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli())
                .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L))
                .build();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> explain(request));
    }

    @RequestMapping(path = "/explainEvaluation", method = RequestMethod.POST)
    @ApiOperation(value = "Explain alert evaluation", response = AlertExplainEvaluationDto.class)
    CompletableFuture<AlertExplainEvaluationDto> explainEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "time", required = false, defaultValue = "") String time,
            @RequestBody AlertDto alert)
    {
        alert.projectId = projectId;
        TExplainEvaluationRequest request = TExplainEvaluationRequest.newBuilder()
                .setAlert(alert.toProto())
                .setEvaluationTimeMillis(time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli())
                .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L))
                .build();
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> explain(request));
    }

    @RequestMapping(path = "/subAlerts/explainEvaluation", method = RequestMethod.POST)
    @ApiOperation(value = "Explain sub-alert evaluation", response = AlertExplainEvaluationDto.class)
    CompletableFuture<AlertExplainEvaluationDto> explainEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "time", required = false, defaultValue = "") String time,
            @RequestBody SubAlertDto subAlert)
    {
        subAlert.projectId = projectId;
        subAlert.parent.projectId = projectId;
        TExplainEvaluationRequest request = TExplainEvaluationRequest.newBuilder()
                .setSubAlert(subAlert.toProto())
                .setEvaluationTimeMillis(time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli())
                .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L))
                .build();
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> explain(request));
    }

    @ApiOperation(value = "Get project stats", response = ProjectStatsDto.class)
    @RequestMapping(path = "/stats", method = RequestMethod.GET)
    CompletableFuture<ProjectStatsDto> getProjectStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId)
    {
        TReadProjectStatsRequest request = TReadProjectStatsRequest.newBuilder()
            .setProjectId(projectId)
            .build();
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> api.readProjectStats(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return ProjectStatsDto.fromProto(response);
            });
    }

    private CompletableFuture<Account> authorize(
            AuthSubject subject,
            String projectId,
            Permission permission,
            String serviceProviderId)
    {
      return authorize(authorizer, subject, projectId, permission, serviceProviderId);
    }

    public static CompletableFuture<Account> authorize(
            Authorizer authorizer,
            AuthSubject subject,
            String projectId,
            Permission permission,
            String serviceProviderId)
    {
        if (!StringUtils.isEmpty(serviceProviderId)) {
            return authorizer.authorize(subject, AuthorizationObject.serviceProvider(serviceProviderId, projectId), Permission.ALERT_MANAGEMENT);
        }
        return authorizer.authorize(subject, projectId, permission);
    }

    private CompletableFuture<AlertExplainEvaluationDto> explain(TExplainEvaluationRequest request) {
        return api.explainEvaluation(request)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertExplainEvaluationDto.fromProto(response);
                });
    }

    public static void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return checkProjectExistence(projectsDao, projectId);
    }

    public static CompletableFuture<Void> checkProjectExistence(ProjectsDao projectsDao, String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }
}
