package ru.yandex.solomon.alert.gateway.endpoint;

import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.dto.alert.CreateAlertsFromTemplateCmd;
import ru.yandex.solomon.alert.gateway.dto.alert.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;

/**
 * @author Alexey Trushkin
 */
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/alerts", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class AlertInternalController {

    private static final String SERVICE_PROVIDER_HEADER = "X-Service-Provider";

    private final ObjectMapper mapper;
    private final ProjectsDao projectsDao;
    private final AlertApi api;
    private final Authorizer authorizer;

    @Autowired
    public AlertInternalController(AlertApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        this.api = api;
        this.projectsDao = projectsDao;
        this.authorizer = authorizer;
        mapper = new ObjectMapper();
        mapper.enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);
    }

    @RequestMapping(method = RequestMethod.POST)
    @ApiOperation(value = "Create new alerts from templates", response = CreateAlertsFromTemplateResponse.class)
    CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody CreateAlertsFromTemplateCmd cmd,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderId)
    {
        var protoRequest = cmd.toProto(projectId, mapper).toBuilder()
                .setCreatedBy(subject.getUniqueId())
                .build();
        return AlertController.authorize(authorizer, subject, projectId, Permission.CONFIGS_GET, serviceProviderId)
                .thenCompose(aVoid -> AlertController.checkProjectExistence(projectsDao, projectId))
                .thenCompose(aVoid -> api.createAlerts(protoRequest))
                .thenApply(response -> {
                    AlertController.ensureStatusValid(response.getRequestStatusCode(), response::getStatusMessage);
                    return CreateAlertsFromTemplateResponse.fromProto(response);
                });
    }
}
