package ru.yandex.solomon.alert.gateway.endpoint;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertFullList;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertLabelsDto;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.EOrderDirection;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class AlertsListController {
    private static final String SERVICE_PROVIDER_HEADER = "X-Service-Provider";

    private final AlertApi api;
    private final Authorizer authorizer;

    @Autowired
    public AlertsListController(AlertApi api, Authorizer authorizer) {
        this.api = api;
        this.authorizer = authorizer;
    }

    @GetMapping("/alertsFullModel")
    @ApiOperation(value = "List alerts full models", response = AlertFullList.class)
    CompletableFuture<AlertFullList> listAlertsByProject(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByStates", required = false) List<EAlertState> filterByState,
            @RequestParam(value = "filterByTypes", required = false) List<EAlertType> filterByType,
            @RequestParam(value = "filterByEvaluationStatus", required = false) List<TEvaluationStatus.ECode> filterByEvaluation,
            @RequestParam(value = "filterByNotificationId", required = false) List<String> filterByNotificationId,
            @RequestParam(value = "filterByNotificationStatus", required = false) List<TNotificationStatus.ECode> filterByNotificationStatus,
            @RequestParam(value = "filterByMuteId", required = false) List<String> filterByMuteId,
            @RequestParam(value = "labelsSelector", required = false, defaultValue = "") String labelsSelector,
            @RequestParam(value = "orderByName", required = false) EOrderDirection orderByName,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken,
            @RequestParam(value = "templateServiceProviderId", defaultValue = "", required = false) String templateServiceProviderId,
            @RequestHeader(value = SERVICE_PROVIDER_HEADER, required = false, defaultValue = "") String serviceProviderHeader)
    {
        TListAlertRequest request = TListAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setPageToken(pageToken)
                .setPageSize(pageSize)
                .setFilterByName(filterByName)
                .addAllFilterByState(Nullables.orEmpty(filterByState))
                .addAllFilterByType(Nullables.orEmpty(filterByType))
                .addAllFilterByEvaluationStatus(Nullables.orEmpty(filterByEvaluation))
                .addAllFilterByNotificationId(Nullables.orEmpty(filterByNotificationId))
                .addAllFilterByNotificationStatus(Nullables.orEmpty(filterByNotificationStatus))
                .setFilterByMuteReference(TListAlertRequest.MuteReference.newBuilder().addAllIds(Nullables.orEmpty(filterByMuteId)))
                .setOrderByName(Nullables.orDefault(orderByName, EOrderDirection.ASC))
                .setFilterCreatedByServiceProvider(serviceProviderHeader)
                .setTemplateProvider(templateServiceProviderId)
                .setLabelsSelector(labelsSelector)
                .setFullResultModel(true)
                .build();

        return authorize(subject, projectId, Permission.CONFIGS_GET, serviceProviderHeader)
                .thenCompose(aVoid -> api.listAlerts(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertFullList.fromProto(response);
                });
    }


    @ApiOperation(value = "Get project label keys in alerts", response = AlertLabelsDto.class)
    @RequestMapping(path = "/alertsLabelKeys", method = RequestMethod.GET)
    CompletableFuture<AlertLabelsDto> getProjectStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId)
    {
        ListAlertLabelsRequest request = ListAlertLabelsRequest.newBuilder()
                .setProjectId(projectId)
                .build();
        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                .thenCompose(aVoid -> api.listAlertLabels(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatusCode(), response::getStatusMessage);
                    return AlertLabelsDto.fromProto(response);
                });
    }

    private CompletableFuture<Account> authorize(
            AuthSubject subject,
            String projectId,
            Permission permission,
            String serviceProviderId)
    {
        if (!StringUtils.isEmpty(serviceProviderId)) {
            return authorizer.authorize(subject, AuthorizationObject.serviceProvider(serviceProviderId, projectId), Permission.ALERT_MANAGEMENT);
        }
        return authorizer.authorize(subject, projectId, permission);
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }
}
