package ru.yandex.solomon.alert.gateway.endpoint;

import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.gateway.dto.notificationChannel.EscalationList;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;

/**
 * @author Alexey Trushkin
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/internal/projects/{projectId}/escalations", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class EscalationInternalController {

    private final NotificationApi api;
    private final Authorizer authorizer;

    public EscalationInternalController(NotificationApi api, Authorizer authorizer) {
        this.api = api;
        this.authorizer = authorizer;
    }

    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "List escalations", response = EscalationList.class)
    CompletableFuture<EscalationList> listEscalation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName) {
        TListEscalationsRequest.Builder builder = TListEscalationsRequest.newBuilder()
                .setProjectId(projectId)
                .setFilterByName(filterByName);

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> api.listEscalations(builder.build()))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return EscalationList.fromProto(response, projectId);
                });
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }
}
