package ru.yandex.solomon.alert.gateway.endpoint;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.MuteApi;
import ru.yandex.solomon.alert.gateway.dto.mute.MuteDto;
import ru.yandex.solomon.alert.gateway.dto.mute.MuteListDto;
import ru.yandex.solomon.alert.gateway.dto.mute.MuteStatsDto;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.MuteStatus;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/mutes", produces = MediaType.APPLICATION_JSON_VALUE)
public class MuteController {
    private final ProjectsDao projectsDao;
    private final MuteApi api;
    private final Authorizer authorizer;

    public MuteController(MuteApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        this.api = api;
        this.projectsDao = projectsDao;
        this.authorizer = authorizer;
    }

    @ApiOperation(value = "Create new mute", response = MuteDto.class)
    @RequestMapping(method = RequestMethod.POST)
    CompletableFuture<MuteDto> createMute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody MuteDto mute)
    {
        var now = Instant.now();
        mute.projectId = projectId;
        mute.createdBy = subject.getUniqueId();
        mute.createdAt = now;
        mute.updatedBy = subject.getUniqueId();
        mute.updatedAt = now;

        var request = CreateMuteRequest.newBuilder()
                .setMute(mute.toProtoBuilder())
                .build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_CREATE)
                .thenCompose(aVoid -> checkProjectExistence(projectId))
                .thenCompose(aVoid -> api.createMute(request, 0))
                .thenApply(resp -> MuteDto.fromProto(resp.getMute()));
    }

    @ApiOperation(value = "Get mute by id", response = MuteDto.class)
    @RequestMapping(path = "/{muteId}", method = RequestMethod.GET)
    CompletableFuture<MuteDto> readMute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("muteId") String muteId)
    {
        var request = ReadMuteRequest.newBuilder()
                .setId(muteId)
                .setProjectId(projectId)
                .build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_GET)
                .thenCompose(aVoid -> api.readMute(request, 0))
                .thenApply(resp -> MuteDto.fromProto(resp.getMute()));
    }

    @ApiOperation(value = "Update mute by id", response = MuteDto.class)
    @RequestMapping(path = "/{muteId}", method = RequestMethod.PUT)
    public CompletableFuture<MuteDto> updateMute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("muteId") String muteId,
            @RequestBody MuteDto mute)
    {
        mute.id = muteId;
        mute.projectId = projectId;
        mute.updatedBy = subject.getUniqueId();
        mute.updatedAt = Instant.now();

        var request = UpdateMuteRequest.newBuilder()
                .setMute(mute.toProtoBuilder())
                .build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_UPDATE)
                .thenCompose(aVoid -> checkProjectExistence(projectId))
                .thenCompose(aVoid -> api.updateMute(request, 0))
                .thenApply(resp -> MuteDto.fromProto(resp.getMute()));
    }

    @ApiOperation(value = "Delete mute by id")
    @RequestMapping(path = "/{muteId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public CompletableFuture<Void> deleteMute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("muteId") String muteId)
    {
        var request = DeleteMuteRequest.newBuilder()
                .setId(muteId)
                .setProjectId(projectId)
                .build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_DELETE)
                .thenCompose(aVoid -> checkProjectExistence(projectId))
                .thenCompose(aVoid -> api.deleteMute(request, 0))
                .thenAccept(ignore -> {});
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new NotFoundException(String.format("project %s does not exist", projectId));
                    }
                });
    }

    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "List mutes", response = MuteListDto.class)
    CompletableFuture<MuteListDto> listMutes(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByStates", required = false) List<MuteStatus> filterByState,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken)
    {
        var builder = ListMutesRequest.newBuilder()
                .setProjectId(projectId)
                .setPageToken(pageToken)
                .setFilterByName(Nullables.orEmpty(filterByName))
                .addAllFilterByStates(Nullables.orEmpty(filterByState))
                .setPageSize(pageSize);

        var request = builder.build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_GET)
                .thenCompose(aVoid -> api.listMutes(request, 0))
                .thenApply(MuteListDto::fromProto);
    }

    @RequestMapping(path = "/stats", method = RequestMethod.GET)
    @ApiOperation(value = "Read mute stats", response = MuteStatsDto.class)
    CompletableFuture<MuteStatsDto> readMuteStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "folderId", defaultValue = "", required = false) String folderId)
    {
        var request = ReadMuteStatsRequest.newBuilder()
                .setProjectId(projectId)
                .setFolderId(Nullables.orEmpty(folderId))
                .build();

        return authorizer.authorize(subject, projectId, Permission.MUTES_GET)
                .thenCompose(aVoid -> api.readMuteStats(request, 0))
                .thenApply(response -> MuteStatsDto.fromProto(response.getMutesStats()));
    }
}
