package ru.yandex.solomon.alert.gateway.endpoint;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.gateway.dto.notificationChannel.NotificationChannelDto;
import ru.yandex.solomon.alert.gateway.dto.notificationChannel.NotificationChannelList;
import ru.yandex.solomon.alert.protobuf.EOrderDirection;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.ENotificationChannelType;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/notificationChannels", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class NotificationChannelController {
    private static final Logger logger = LoggerFactory.getLogger(NotificationChannelController.class);

    private final NotificationApi api;
    private final ProjectsDao projectsDao;
    private final Authorizer authorizer;

    @Autowired
    public NotificationChannelController(NotificationApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        this.api = api;
        this.projectsDao = projectsDao;
        this.authorizer = authorizer;
    }

    @ApiOperation(value = "Get notification channel by id", response = NotificationChannelDto.class)
    @RequestMapping(path = "/{notificationChannelId}", method = RequestMethod.GET)
    CompletableFuture<NotificationChannelDto> getNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("notificationChannelId") String notificationChannelId)
    {
        TReadNotificationRequest request = TReadNotificationRequest.newBuilder()
            .setNotificationId(notificationChannelId)
            .setProjectId(projectId)
            .build();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
            .thenCompose(aVoid -> api.readNotification(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelDto.fromProto(response.getNotification());
            });
    }

    @ApiOperation(value = "Update notification channel by id", response = NotificationChannelDto.class)
    @RequestMapping(path = "/{notificationChannelId}", method = RequestMethod.PUT)
    public CompletableFuture<NotificationChannelDto> updateNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("notificationChannelId") String notificationChannelId,
            @RequestBody NotificationChannelDto notification)
    {
        notification.id = notificationChannelId;
        notification.projectId = projectId;
        notification.updatedBy = subject.getUniqueId();
        notification.updatedAt = Instant.now().toString();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_UPDATE)
            .thenCompose(aVoid -> checkProjectExistence(projectId))
            .thenCompose(aVoid -> api.updateNotification(TUpdateNotificationRequest.newBuilder()
                .setNotification(notification.toProto())
                .build()))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelDto.fromProto(response.getNotification());
            });
    }

    @RequestMapping(method = RequestMethod.POST)
    @ApiOperation(value = "Create new notification channel", response = NotificationChannelDto.class)
    CompletableFuture<NotificationChannelDto> createNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestBody NotificationChannelDto notification)
    {
        String now = Instant.now().toString();
        notification.projectId = projectId;
        notification.createdBy = subject.getUniqueId();
        notification.createdAt = now;
        notification.updatedBy = subject.getUniqueId();
        notification.updatedAt = now;

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_CREATE)
            .thenCompose(aVoid -> checkProjectExistence(projectId))
            .thenCompose(aVoid -> api.createNotification(TCreateNotificationRequest.newBuilder()
                .setNotification(notification.toProto())
                .build()))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelDto.fromProto(response.getNotification());
            });
    }

    @RequestMapping(path = "/{notificationChannelId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @ApiOperation(value = "Delete notification")
    CompletableFuture<Void> deleteNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("notificationChannelId") String notificationChannelId)
    {
        TDeleteNotificationRequest request = TDeleteNotificationRequest.newBuilder()
            .setNotificationId(notificationChannelId)
            .setProjectId(projectId)
            .build();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_DELETE)
            .thenCompose(aVoid -> api.deleteNotification(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return null;
            });
    }

    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "List notification channels", response = NotificationChannelList.class)
    CompletableFuture<NotificationChannelList> listNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByType", required = false) List<ENotificationChannelType> filterByType,
            @RequestParam(value = "filterByDefaultSeverity", required = false) List<Severity> filterByDefaultSeverity,
            @RequestParam(value = "orderByName", required = false) EOrderDirection orderByName,
            @RequestParam(value = "orderByType", required = false) EOrderDirection orderByType,
            @RequestParam(value = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken)
    {
        TListNotificationsRequest.Builder builder = TListNotificationsRequest.newBuilder()
                .setProjectId(projectId)
                .setPageToken(pageToken)
                .setPageSize(pageSize)
                .setFilterByName(filterByName)
                .addAllFilterByType(Nullables.orEmpty(filterByType))
                .addAllFilterByDefaultSeverity(Nullables.orEmpty(filterByDefaultSeverity));

        if (orderByName != null) {
            builder.setOrderByName(orderByName);
        }

        if (orderByType != null) {
            builder.setOrderByType(orderByType);
        }

        TListNotificationsRequest request = builder.build();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
            .thenCompose(aVoid -> api.listNotification(request))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelList.fromProto(response);
            });
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", projectId));
                }
            });
    }
}
