package ru.yandex.solomon.alert.gateway.endpoint;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.ServiceProviderResourceSeverity;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.gateway.dto.notificationChannel.NotificationChannelsForSeverityResponse;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;

/**
 * @author Alexey Trushkin
 */
@Api(tags = {"alerting"})
@RestController
@RequestMapping(path = "/api/v2/projects/{projectId}/notificationChannelsForSeverity", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class NotificationChannelForSeverityController {
    private final NotificationApi api;
    private final Authorizer authorizer;

    @Autowired
    public NotificationChannelForSeverityController(NotificationApi api, Authorizer authorizer) {
        this.api = api;
        this.authorizer = authorizer;
    }

    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "List notification channels for severity", response = NotificationChannelsForSeverityResponse.class)
    CompletableFuture<NotificationChannelsForSeverityResponse> listNotification(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "severity", defaultValue = "SERVICE_PROVIDER_RESOURCE_SEVERITY_UNSPECIFIED", required = false) ServiceProviderResourceSeverity severity)
    {
        if (severity.getNumber() < 1) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("severity parameter must be specified"));
        }
        TListNotificationsRequest.Builder builder = TListNotificationsRequest.newBuilder()
                .setProjectId(projectId)
                .setPageSize(1000)
                .addAllFilterByDefaultSeverity(List.of(Severity.SEVERITY_CRITICAL, Severity.SEVERITY_DISASTER, Severity.SEVERITY_INFO));

        TListNotificationsRequest request = builder.build();

        return authorizer.authorize(subject, projectId, Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> api.listNotification(request))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return NotificationChannelsForSeverityResponse.fromProto(severity, response);
                });
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }
}
