package ru.yandex.solomon.alert.gateway.endpoint;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.bolts.collection.Try;
import ru.yandex.solomon.alert.gateway.converter.DashboardYasmQueryConverter;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertAlertResultDto;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertAlertsRequestDto;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertAlertsResponseDto;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertYasmQueriesDto;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertYasmQueriesResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.yasm.alert.converter.YasmAlertConverter;

/**
 * @author Ivan Tsybulin
 */
@Api(tags = {"Yasm"})
@RestController
@RequestMapping(path = "/api/v2/", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class YasmController {
    private final Authorizer authorizer;
    private final YasmAlertConverter converter;
    private final DashboardYasmQueryConverter dashboardYasmQueryConverter;

    @Autowired
    public YasmController(Authorizer authorizer, DashboardYasmQueryConverter dashboardYasmQueryConverter) {
        this.authorizer = authorizer;
        this.dashboardYasmQueryConverter = dashboardYasmQueryConverter;
        this.converter = new YasmAlertConverter("yasm_");
    }

    @ApiOperation(value = "Validate and convert Yasm alerts", response = ConvertAlertsResponseDto.class)
    @RequestMapping(path = "/yasmAlerts/convert", method = RequestMethod.POST)
    public CompletableFuture<ConvertAlertsResponseDto> convert(
            @RequireAuth AuthSubject subject,
            @RequestBody ConvertAlertsRequestDto request)
    {
        return CompletableFuture.completedFuture(new ConvertAlertsResponseDto(
                request.alerts.stream()
                        .map(alert -> Try.tryCatchException(() -> converter.convertAlert(alert).build()))
                        .map(ConvertAlertResultDto::fromConversionResult)
                        .collect(Collectors.toList())
        ));
    }

    @ApiOperation(value = "Validate and convert Yasm queries", response = ConvertYasmQueriesResponse.class)
    @RequestMapping(path = "/yasmQuery/convert", method = RequestMethod.POST)
    public CompletableFuture<ConvertYasmQueriesResponse> convertQuery(
            @RequireAuth AuthSubject subject,
            @RequestBody ConvertYasmQueriesDto request)
    {
         return CompletableFuture.completedFuture(new ConvertYasmQueriesResponse(dashboardYasmQueryConverter.prepareUrl(request)));
    }
}
