package ru.yandex.solomon.alert.gateway.dto.alert;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.alert.protobuf.AlertFromTemplate;
import ru.yandex.solomon.alert.protobuf.AlertParameter;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TExpression;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class AlertDtoTest {
    @Parameterized.Parameter
    public TAlert.TypeCase type;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object> data() {
        return Arrays.asList(TAlert.TypeCase.EXPRESSION, TAlert.TypeCase.THRESHOLD, TAlert.TypeCase.ALERT_FROM_TEMPLATE);
    }

    @Test
    public void protoDtoAndBack() {
        TAlert source = randomAlert(type);
        AlertDto dto = toDto(source);
        TAlert restoreProto = toProto(dto);

        assertThat(restoreProto, equalTo(source));
    }

    @Test
    public void protoDtoJsonAndBack() throws Exception {
        TAlert source = randomAlert(type);
        AlertDto dto = toDto(source);
        String json = toJson(dto);
        AlertDto restoreDto = toDto(json);
        TAlert restoreProto = toProto(restoreDto);

        assertThat("Proto: " + restoreDto + "\n" +
                        "Dto: " + dto + "\n" +
                        "Json: " + json,
                restoreProto, equalTo(source));
    }

    @Test
    public void newChannelFieldsArePreferred() throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        var alert = mapper.readValue(load("solomon-uptime-new.json"), AlertDto.class).toProto();

        assertThat(alert.getNotificationChannelIdsCount(), greaterThan(0));
        assertThat(alert.getConfiguredNotificationChannelsCount(), greaterThan(0));
    }

    private String load(String res) {
        try {
            URL url = getClass().getResource(res);
            return Resources.toString(url, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private TAlert randomAlert(TAlert.TypeCase type) {
        ThreadLocalRandom random = ThreadLocalRandom.current();

        TAlert.Builder builder = TAlert.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setName("Name with random: " + random.nextInt())
                .setDescription("Description with random: " + random.nextInt())
                .setProjectId(UUID.randomUUID().toString())
                .setVersion(random.nextInt(0, 1000))
                .setUpdatedAt(Instant.now().plusMillis(random.nextLong(0, 1_000_000)).toEpochMilli())
                .setCreatedAt(Instant.now().minusMillis(random.nextLong(0, 1_000_000)).toEpochMilli())
                .setState(EAlertState.values()[random.nextInt(0, EAlertState.values().length - 1)])
                .setSeverity(Severity.values()[random.nextInt(0, Severity.values().length - 1)])
                .addNotificationChannelIds("notify-" + UUID.randomUUID().toString())
                .addNotificationChannelIds("notify - 2" + UUID.randomUUID().toString())
                .setPeriodMillis(ThreadLocalRandom.current().nextLong(1000, 1_000_00))
                .setDelaySeconds(ThreadLocalRandom.current().nextInt(0, 100))
                .addAllGroupByLabels(random.nextBoolean() ? Collections.emptyList() : Collections.singleton("host"))
                .putAllLabels(random.nextBoolean() ? Map.of() : Map.of("lbl1", "val1"))
                .putAllAnnotations(random.nextBoolean() ? Map.of() : Map.of("k1", "v1"))
                .putAllServiceProviderAnnotations(random.nextBoolean() ? Map.of() : Map.of("k2", "v2"))
                .addAllEscalations(List.of("e1", "e2"));

        switch (type) {
            case EXPRESSION:
                builder.setExpression(randomExpression(random));
                break;
            case THRESHOLD:
                builder.setThreshold(randomThreshold(random));
                break;
            case ALERT_FROM_TEMPLATE:
                builder.setAlertFromTemplate(randomAlertFromTemplate(random));
                break;
            default:
                throw new UnsupportedOperationException("Unsupported alert type: " + type);
        }

        return builder.build();
    }

    private TThreshold randomThreshold(ThreadLocalRandom random) {
        Selectors selectors = Selectors.parse("project=solomon, cluster=local, service=test, sensor=idleTime, host=solomon-"
                        + random.nextInt(1, 100));

        return TThreshold.newBuilder()
                .setThreshold(random.nextDouble())
                .setComparison(ECompare.values()[random.nextInt(0, ECompare.values().length - 1)])
                .setThresholdType(EThresholdType.values()[random.nextInt(0, EThresholdType.values().length - 1)])
                .setNewSelectors(LabelSelectorConverter.selectorsToNewProto(selectors))
                .setTransformations(random.nextBoolean() ? "" : "derivative")
                .addAllPredicateRules(IntStream.range(0, random.nextInt(2, 5))
                        .mapToObj((ignore) -> randomPredicateRule(random))
                        .collect(Collectors.toList()))
                .build();
    }

    private TPredicateRule randomPredicateRule(ThreadLocalRandom random) {
        return TPredicateRule.newBuilder()
                .setThreshold(random.nextDouble())
                .setComparison(ECompare.values()[random.nextInt(0, ECompare.values().length - 1)])
                .setThresholdType(EThresholdType.values()[random.nextInt(0, EThresholdType.values().length - 1)])
                .setTargetStatus(TPredicateRule.ETargetStatus.values()[random.nextInt(1, TPredicateRule.ETargetStatus.values().length - 1)])
                .build();
    }

    private TExpression randomExpression(ThreadLocalRandom random) {
        return TExpression.newBuilder()
                .setProgram("let rr = random01() < " + random.nextDouble(1, 10) + ";")
                .setCheckExpression("rr")
                .build();
    }

    private AlertFromTemplate randomAlertFromTemplate(ThreadLocalRandom random) {
        return AlertFromTemplate.newBuilder()
                .setTemplateId("id" + random.nextInt())
                .setTemplateVersionTag("idTag" + random.nextInt())
                .addAllAlertThresholds(getThresholds())
                .addAllAlertParameters(getParameters())
                .build();
    }

    private List<AlertParameter> getParameters() {
        return List.of(
                AlertParameter.newBuilder()
                        .setDoubleParameterValue(AlertParameter.DoubleParameterValue.newBuilder()
                                .setName("DoubleParameterValue param")
                                .setValue(2.1)
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setIntegerParameterValue(AlertParameter.IntegerParameterValue.newBuilder()
                                .setName("IntegerParameterValue param")
                                .setValue(22)
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setTextParameterValue(AlertParameter.TextParameterValue.newBuilder()
                                .setName("TextParameterValue param")
                                .setValue("text param")
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setTextListParameterValue(AlertParameter.TextListParameterValue.newBuilder()
                                .setName("TextParameterValues param")
                                .addAllValues(List.of("1", "2", "3"))
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setLabelListParameterValue(AlertParameter.LabelListParameterValue.newBuilder()
                                .setName("LabelParameterValues param")
                                .addAllValues(List.of("1l", "2l", "3L"))
                                .build())
                        .build()
        );
    }

    private List<AlertParameter> getThresholds() {
        return List.of(
                AlertParameter.newBuilder()
                        .setDoubleParameterValue(AlertParameter.DoubleParameterValue.newBuilder()
                                .setName("DoubleParameterValue")
                                .setValue(1.1)
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setIntegerParameterValue(AlertParameter.IntegerParameterValue.newBuilder()
                                .setName("IntegerParameterValue")
                                .setValue(2)
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setTextParameterValue(AlertParameter.TextParameterValue.newBuilder()
                                .setName("TextParameterValue")
                                .setValue("text")
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setTextListParameterValue(AlertParameter.TextListParameterValue.newBuilder()
                                .setName("TextParameterValues")
                                .addAllValues(List.of("1", "2"))
                                .build())
                        .build(),
                AlertParameter.newBuilder()
                        .setLabelListParameterValue(AlertParameter.LabelListParameterValue.newBuilder()
                                .setName("LabelParameterValues")
                                .addAllValues(List.of("1l", "2l"))
                                .build())
                        .build()
        );
    }

    private AlertDto toDto(TAlert proto) {
        return AlertDto.fromProto(proto);
    }

    private TAlert toProto(AlertDto dto) {
        return dto.toProto();
    }

    private String toJson(AlertDto dto) throws JsonProcessingException {
        return getMapper().writerWithDefaultPrettyPrinter().writeValueAsString(dto);
    }

    private AlertDto toDto(String json) throws IOException {
        return getMapper().readValue(json, AlertDto.class);
    }

    private ObjectMapper getMapper() {
        return new ObjectMapper();
    }

}
