package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monitoring.api.v3.UnitFormat;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplateParameter;
import ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.Expression;
import ru.yandex.solomon.alert.protobuf.IntegerTemplateParameter;
import ru.yandex.solomon.alert.protobuf.LabelListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.PredicateRule;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TextListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.TextTemplateParameter;
import ru.yandex.solomon.alert.protobuf.Threshold;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author Alexey Trushkin
 */
@RunWith(Parameterized.class)
public class AlertTemplateDtoTest {

    @Parameterized.Parameter
    public AlertTemplate.TypeCase type;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object> data() {
        return Arrays.asList(AlertTemplate.TypeCase.EXPRESSION, AlertTemplate.TypeCase.THRESHOLD);
    }

    @Test
    public void protoDtoAndBack() {
        var source = randomAlertTemplate(type);
        AlertTemplateDto dto = toDto(source);
        AlertTemplate restoreProto = toProto(dto);

        assertThat(restoreProto, equalTo(source));
    }

    private AlertTemplate randomAlertTemplate(AlertTemplate.TypeCase type) {
        ThreadLocalRandom random = ThreadLocalRandom.current();

        AlertTemplate.Builder builder = AlertTemplate.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setTemplateVersionTag(UUID.randomUUID().toString())
                .setServiceProviderId(UUID.randomUUID() + " service provider")
                .setName("Name with random: " + random.nextInt())
                .setDescription("Description with random: " + random.nextInt())
                .setCreatedBy("created by")
                .setModifiedBy("modified by")
                .setPeriodMillis(ThreadLocalRandom.current().nextLong(1000, 1_000_00))
                .setDelaySeconds(ThreadLocalRandom.current().nextInt(0, 100))
                .putAllAnnotations(random.nextBoolean() ? Map.of() : Map.of("key", "host"))
                .putAllLabels(random.nextBoolean() ? Map.of() : Map.of("key2", "host2"))
                .addAllGroupByLabels(random.nextBoolean() ? Collections.emptyList() : Collections.singleton("host"))
                .setNoPointsPolicy(NoPointsPolicy.NO_POINTS_NO_DATA)
                .setResolvedEmptyPolicy(ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT)
                .setSeverity(Severity.SEVERITY_UNSPECIFIED)
                .addAllAlertTemplateParameters(getParameters())
                .addAllAlertTemplateThresholds(getThresholds())
                .setDefault(random.nextBoolean());

        switch (type) {
            case EXPRESSION:
                builder.setExpression(randomExpression(random));
                break;
            case THRESHOLD:
                builder.setThreshold(randomThreshold(random));
                break;
            default:
                throw new UnsupportedOperationException("Unsupported alert type: " + type);
        }

        return builder.build();
    }

    private Threshold randomThreshold(ThreadLocalRandom random) {
        var selectors = "project=solomon, cluster=local, service=test, sensor=idleTime, host=solomon-"
                + random.nextInt(1, 100);

        return Threshold.newBuilder()
                .setSelectors(selectors)
                .setTransformations(random.nextBoolean() ? "" : "derivative")
                .addAllPredicateRules(IntStream.range(0, random.nextInt(2, 5))
                        .mapToObj((ignore) -> randomPredicateRule(random))
                        .collect(Collectors.toList()))
                .build();
    }

    private PredicateRule randomPredicateRule(ThreadLocalRandom random) {
        return PredicateRule.newBuilder()
                .setThresholdParameterTemplate(random.nextInt() + "")
                .setThreshold(random.nextDouble())
                .setComparison(ECompare.values()[random.nextInt(0, ECompare.values().length - 1)])
                .setThresholdType(EThresholdType.values()[random.nextInt(0, EThresholdType.values().length - 1)])
                .setTargetStatus(TPredicateRule.ETargetStatus.values()[random.nextInt(1, TPredicateRule.ETargetStatus.values().length - 1)])
                .build();
    }

    private Expression randomExpression(ThreadLocalRandom random) {
        return Expression.newBuilder()
                .setProgram("let rr = random01() < " + random.nextDouble(1, 10) + ";")
                .build();
    }

    private List<AlertTemplateParameter> getParameters() {
        return List.of(
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue param")
                                .setTitle("DoubleParameterValue param title")
                                .setDescription("DoubleParameterValue param descr")
                                .setDefaultValue(2.1)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue param2")
                                .setTitle("DoubleParameterValue param title2")
                                .setDescription("DoubleParameterValue param descr2")
                                .setDefaultValue(2.12)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setIntegerParameter(IntegerTemplateParameter.newBuilder()
                                .setName("IntegerParameterValue param")
                                .setTitle("IntegerParameterValue param title")
                                .setDescription("IntegerParameterValue param descr")
                                .setDefaultValue(22)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextParameter(TextTemplateParameter.newBuilder()
                                .setName("TextParameterValue param")
                                .setTitle("TextParameterValue param title")
                                .setDescription("TextParameterValue param descr")
                                .setDefaultValue("text param")
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextListParameter(TextListTemplateParameter.newBuilder()
                                .setName("TextParameterValues param")
                                .setTitle("TextParameterValues param title")
                                .setDescription("TextParameterValues param descr")
                                .addAllDefaultValues(List.of("1", "2", "3"))
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setLabelListParameter(LabelListTemplateParameter.newBuilder()
                                .setName("LabelParameterValues param")
                                .setTitle("LabelParameterValues param title")
                                .setDescription("LabelParameterValues param descr")
                                .setSelectors("selector")
                                .setLabelKey("key")
                                .setProjectId("prj")
                                .setMultiselectable(false)
                                .addAllDefaultValues(List.of("1l", "2l", "3L"))
                                .build())
                        .build()
        );
    }

    private List<AlertTemplateParameter> getThresholds() {
        return List.of(
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue threshold")
                                .setTitle("DoubleParameterValue threshold title")
                                .setDescription("DoubleParameterValue threshold descr")
                                .setDefaultValue(12.1)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue threshold2")
                                .setTitle("DoubleParameterValue threshold title2")
                                .setDescription("DoubleParameterValue threshold descr2")
                                .setDefaultValue(12.12)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setIntegerParameter(IntegerTemplateParameter.newBuilder()
                                .setName("IntegerParameterValue threshold")
                                .setTitle("IntegerParameterValue threshold title")
                                .setDescription("IntegerParameterValue threshold descr")
                                .setDefaultValue(122)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextParameter(TextTemplateParameter.newBuilder()
                                .setName("TextParameterValue threshold")
                                .setTitle("TextParameterValue threshold title")
                                .setDescription("TextParameterValue threshold descr")
                                .setDefaultValue("1text threshold")
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextListParameter(TextListTemplateParameter.newBuilder()
                                .setName("TextParameterValues threshold")
                                .setTitle("TextParameterValues threshold title")
                                .setDescription("TextParameterValues threshold descr")
                                .addAllDefaultValues(List.of("0", "1", "2", "3"))
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setLabelListParameter(LabelListTemplateParameter.newBuilder()
                                .setName("LabelParameterValues threshold")
                                .setTitle("LabelParameterValues threshold title")
                                .setDescription("LabelParameterValues threshold descr")
                                .setSelectors("1selector")
                                .setLabelKey("1key")
                                .setProjectId("prj")
                                .setMultiselectable(true)
                                .addAllDefaultValues(List.of("0L", "1l", "2l", "3L"))
                                .build())
                        .build()
        );
    }

    private AlertTemplateDto toDto(AlertTemplate proto) {
        return AlertTemplateDto.fromProto(proto);
    }

    private AlertTemplate toProto(AlertTemplateDto dto) {
        return dto.toProto();
    }
}
