package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.core.exceptions.BadRequestException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Alexey Trushkin
 */
public class CreateAlertsFromTemplateCmdTest {

    private static final String EXAMPLE = "\"supplier\",\"type\",\"consumer\",\"consumer_id\",\"consumer_slug\",\"tags\",\"attributes\",\"modified_at\",\"state\",\"has_monitoring\",\"need_monitoring\"\n" +
            "\"Solomon\",\"Проект\",\"Zora\",\"36\",\"fetcher\",\"production\",\"ID:None;name:projectId:zora;resource_id:{'projectId': 'zora'};responsible:zosimov;resource_type:project;monitoring_stats:{'alerts_status': {'Shard metrics in response quota usage(monitoring-shards-url-quota)': 'false', 'Shard metrics quota usage(monitoring-metrics-quota)': 'false', 'Shard fetch errors(monitoring-fetch-errors)': 'false'}};service_provider_id:monitoring\",\"2022-03-09T15:41:32.235489Z\",\"granted\",\"False\",\"True\"\n";
    private static final String NEW_FORMAT_EXAMPLE = "projectId:QUESTIONAPPIOS,resourceType:project\nprojectId:QUESTIONAPPIOS2,resourceType:mon1";

    private ObjectMapper mapper;

    @Before
    public void setUp() {
        mapper = new ObjectMapper();
        mapper.enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);
    }

    @Test(expected = BadRequestException.class)
    public void cmdToProto_sp() {
        new CreateAlertsFromTemplateCmd().toProto("1", mapper);
    }

    @Test(expected = BadRequestException.class)
    public void cmdToProto_templates() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "sp";
        cmd.toProto("1", mapper);
    }

    @Test(expected = BadRequestException.class)
    public void cmdToProto_channels() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "sp";
        cmd.templateIds = Set.of("11");
        cmd.toProto("1", mapper);
    }

    @Test(expected = BadRequestException.class)
    public void cmdToProto_rows() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "sp";
        cmd.templateIds = Set.of("11");
        cmd.channels = List.of(AssociatedChannelDto.of("1", null));
        cmd.toProto("1", mapper);
    }

    @Test(expected = BadRequestException.class)
    public void cmdToProto_rows_error() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "monitoring1";
        cmd.templateIds = Set.of("11", "22");
        cmd.channels = List.of(AssociatedChannelDto.of("1", new ChannelConfigDto()));
        cmd.rowsText = EXAMPLE;
        cmd.toProto("1", mapper);
    }

    @Test
    public void cmdToProto() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "monitoring";
        cmd.templateIds = Set.of("11", "22");
        cmd.channels = List.of(AssociatedChannelDto.of("1", new ChannelConfigDto()));
        cmd.escalations = List.of("1234");
        cmd.rowsText = EXAMPLE;
        cmd.format = "csv";
        var proto = cmd.toProto("project", mapper);
        assertEquals(cmd.serviceProviderId, proto.getServiceProviderId());
        assertEquals(new ArrayList<>(cmd.templateIds), proto.getTemplateIdsList());
        assertEquals(new ArrayList<>(cmd.escalations), proto.getEscalationsList());
        assertEquals(List.of(CreateAlertsFromTemplateRequest.Resource.newBuilder()
                .putAllResourceParameters(Map.of("projectId", "zora", "resourceType", "project"))
                .build()), proto.getResourcesList());
        assertTrue(proto.getChannelsCount() > 0);
    }

    @Test
    public void cmdToProto_newFormat() {
        var cmd = new CreateAlertsFromTemplateCmd();
        cmd.serviceProviderId = "monitoring";
        cmd.templateIds = Set.of("11", "22");
        cmd.channels = List.of(AssociatedChannelDto.of("1", new ChannelConfigDto()));
        cmd.escalations = List.of("1234");
        cmd.rowsText = NEW_FORMAT_EXAMPLE;
        var proto = cmd.toProto("project", mapper);
        assertEquals(cmd.serviceProviderId, proto.getServiceProviderId());
        assertEquals(new ArrayList<>(cmd.templateIds), proto.getTemplateIdsList());
        assertEquals(new ArrayList<>(cmd.escalations), proto.getEscalationsList());
        assertEquals(List.of(CreateAlertsFromTemplateRequest.Resource.newBuilder()
                        .putAllResourceParameters(Map.of("projectId", "QUESTIONAPPIOS", "resourceType", "project"))
                        .build(),
                CreateAlertsFromTemplateRequest.Resource.newBuilder()
                        .putAllResourceParameters(Map.of("projectId", "QUESTIONAPPIOS2", "resourceType", "mon1"))
                        .build()), proto.getResourcesList());
        assertTrue(proto.getChannelsCount() > 0);
    }
}
