package ru.yandex.solomon.alert.gateway.dto.notificationChannel;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Sets;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.notification.PhoneType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudPushType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TDatalensEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TEmailType;
import ru.yandex.solomon.alert.protobuf.notification.THeader;
import ru.yandex.solomon.alert.protobuf.notification.TJugglerType;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.alert.protobuf.notification.TSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TTelegramType;
import ru.yandex.solomon.alert.protobuf.notification.TWebhookType;
import ru.yandex.solomon.alert.protobuf.notification.TYaChatsType;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class NotificationChannelDtoTest {

    @Parameterized.Parameter
    public TNotification.TypeCase type;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object> data() {
        return new ArrayList<>(EnumSet.complementOf(EnumSet.of(TNotification.TypeCase.TYPE_NOT_SET)));
    }

    @Test
    public void protoDtoAndBack() throws Exception {
        TNotification source = randomNotification(type);
        NotificationChannelDto dto = toDto(source);
        TNotification restoreProto = toProto(dto);

        assertThat(restoreProto, equalTo(source));
    }

    @Test
    public void protoDtoJsonAndBack() throws Exception {
        TNotification source = randomNotification(type);
        NotificationChannelDto dto = toDto(source);
        String json = toJson(dto);
        NotificationChannelDto restoreDto = toDto(json);
        TNotification restoreProto = toProto(restoreDto);

        assertThat("Proto: " + restoreDto + "\n" +
                        "Dto: " + dto + "\n" +
                        "Json: " + json,
                restoreProto, equalTo(source));
    }

    private String toJson(NotificationChannelDto dto) throws JsonProcessingException {
        return getMapper().writerWithDefaultPrettyPrinter().writeValueAsString(dto);
    }

    private NotificationChannelDto toDto(String json) throws IOException {
        return getMapper().readValue(json, NotificationChannelDto.class);
    }

    private ObjectMapper getMapper() {
        return new ObjectMapper();
    }

    private TNotification toProto(NotificationChannelDto dto) {
        return dto.toProto();
    }

    private NotificationChannelDto toDto(TNotification proto) {
        return NotificationChannelDto.fromProto(proto);
    }

    private String getHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            return "localhost";
        }
    }

    private TJugglerType randomJuggler(ThreadLocalRandom random) {
        List<String> hosts = Arrays.asList("solomon-stp-man", "solomon-stp-myt", "", "solomon-alert-{{alert.id}}");
        List<String> services = Arrays.asList("my-alert-check", "", "{{alert.id}}");
        List<String> descriptions = Arrays.asList("", "[{{status.code}}]: {{alert.id}}", "{{alert.name}}", "все пропало!");

        Set<String> tags = Sets.newHashSet(
                "solomon-alert",
                "{{alert.id}}",
                UUID.randomUUID().toString(),
                getHostName());
        tags.removeIf(s -> random.nextBoolean());

        Set<TEvaluationStatus.ECode> notifyAbout =
                EnumSet.complementOf(EnumSet.of(TEvaluationStatus.ECode.UNRECOGNIZED));

        notifyAbout.removeIf(status -> random.nextBoolean());

        return TJugglerType.newBuilder()
                .setHost(hosts.get(random.nextInt(hosts.size())))
                .setService(services.get(random.nextInt(services.size())))
                .setDescription(descriptions.get(random.nextInt(descriptions.size())))
                .addAllTags(tags)
                .build();
    }

    private TNotification randomNotification(TNotification.TypeCase type) {
        ThreadLocalRandom random = ThreadLocalRandom.current();

        Set<TEvaluationStatus.ECode> notifyAbout =
                EnumSet.complementOf(EnumSet.of(TEvaluationStatus.ECode.UNRECOGNIZED));

        notifyAbout.removeIf(status -> random.nextBoolean());

        Set<Severity> severities =
                EnumSet.complementOf(EnumSet.of(Severity.UNRECOGNIZED));

        severities.removeIf(s -> random.nextBoolean());

        String id = UUID.randomUUID().toString();
        TNotification.Builder builder = TNotification.newBuilder()
                .setId(id)
                .setProjectId(UUID.randomUUID().toString())
                .setName("Name with random: " + random.nextInt())
                .setUpdatedAt(random.nextLong(0, 1_000_000))
                .setCreatedAt(random.nextLong(0, 1_000_000))
                .setVersion(random.nextInt(0, 1000))
                .addAllNotifyAboutStatuses(notifyAbout)
                .setDefaultForProject(!severities.isEmpty())
                .addAllDefaultForAlertSeverity(severities)
                ;

        switch (type) {
            case JUGGLER -> builder.setJuggler(randomJuggler(random));
            case WEBKOOK -> builder.setWebkook(randomWebhook(random));
            case EMAIL -> builder.setEmail(randomEmail(random));
            case SMS -> builder.setSms(randomSms(random));
            case TELEGRAM -> builder.setTelegram(randomTelegram(random));
            case CLOUDEMAIL -> builder.setCloudEmail(randomCloudEmail(random));
            case CLOUDSMS -> builder.setCloudSms(randomCloudSms(random));
            case YACHATS -> builder.setYaChats(randomYaChats(random));
            case DATALENSEMAIL -> builder.setDatalensEmail(randomDatalensEmail(random));
            case CLOUDPUSH -> builder.setCloudPush(randomCloudPush(random));
            case PHONE -> builder.setPhone(randomPhone(random));
            default -> throw new UnsupportedOperationException("Unknown type: " + type);
        }

        return builder.build();
    }

    private TEmailType randomEmail(ThreadLocalRandom random) {
        Set<String> recipients = Sets.newHashSet(
                "valarie.coffey@yandex-team.ru",
                "frank.osborne@yandex-team.ru",
                "mary.phillis@gmail.com",
                "karin.ruiz@yandex.ru");

        recipients.removeIf(s -> random.nextBoolean());
        if (recipients.isEmpty()) {
            recipients.add("solomon-test@yandex-team.ru");
        }

        return TEmailType.newBuilder()
                .addAllRecipients(recipients)
                .setSubjectTemplate(random.nextBoolean() ? "[{{status.code}}]: {{alert.id}}" : "{{alert.name}}")
                .setContentTemplate(random.nextBoolean() ? "Everything is broken" : "{{status.code}}")
                .build();
    }

    private TCloudEmailType randomCloudEmail(ThreadLocalRandom random) {
        return TCloudEmailType.newBuilder()
            .addRecipients(UUID.randomUUID().toString())
            .build();
    }

    private TCloudSmsType randomCloudSms(ThreadLocalRandom random) {
        return TCloudSmsType.newBuilder()
            .addRecipients(UUID.randomUUID().toString())
            .build();
    }

    private TCloudPushType randomCloudPush(ThreadLocalRandom random) {
        return TCloudPushType.newBuilder()
                .addRecipients(UUID.randomUUID().toString())
                .build();
    }


    public static PhoneType randomPhone(ThreadLocalRandom random) {
        var builder = PhoneType.newBuilder();
        if (random.nextBoolean()) {
            builder
                    .setLogin(UUID.randomUUID().toString());
        } else {
            builder
                    .setDuty(PhoneType.Duty.newBuilder()
                            .setAbcService(UUID.randomUUID().toString())
                            .setDutySlug(UUID.randomUUID().toString())
                            .build());
        }
        return builder.build();
    }

    private TWebhookType randomWebhook(ThreadLocalRandom random) {
        final String tempate;
        if (random.nextBoolean()) {
            tempate = "{\"alertId\": \"{{alert.id}}\", \"status\": \"{{status.code}}\", \"when\": \"{{since}}\"}";
        } else {
            tempate = "{\"alertId\": \"{{alert.id}}\"}";
        }


        return TWebhookType.newBuilder()
                .setUrl("http://localhost:8181/alert/" + random.nextInt())
                .addHeaders(THeader.newBuilder()
                        .setName("Expect")
                        .setValue(String.valueOf(random.nextInt()))
                        .build())
                .setTemplate(tempate)
                .build();
    }

    private TTelegramType randomTelegram(ThreadLocalRandom random) {
        final String tempate;
        if (random.nextBoolean()) {
            tempate = "{\"alertId\": \"{{alert.id}}\", \"status\": \"{{status.code}}\", \"when\": \"{{since}}\"}";
        } else {
            tempate = "{\"alertId\": \"{{alert.id}}\"}";
        }

        boolean isGroup = random.nextBoolean();

        TTelegramType.Builder builder = TTelegramType.newBuilder()
            .setTextTemplate(tempate);
        if (isGroup) {
            builder.setGroupTitle("Solomon-Group");
        } else {
            builder.setLogin("alexlovkov");
        }
        return builder.build();
    }

    private TSmsType randomSms(ThreadLocalRandom random) {
        final String text;
        if (random.nextBoolean()) {
            text = "Everything is broken in {{alertId}}: {{status}}";
        } else {
            text = "{{alertId}}: {{status}}";
        }

        final StringBuilder phoneBuilder = new StringBuilder().append('+');

        for (int i = 0; i < 11; ++i) {
            phoneBuilder.append(random.nextInt(10));
        }

        String phone = phoneBuilder.toString();

        return TSmsType.newBuilder()
            .setPhone(phone)
            .setTextTemplate(text)
            .build();
    }

    private TYaChatsType randomYaChats(ThreadLocalRandom random) {
        final String tempate;
        if (random.nextBoolean()) {
            tempate = "{\"alertId\": \"{{alert.id}}\", \"status\": \"{{status.code}}\", \"when\": \"{{since}}\"}";
        } else {
            tempate = "{\"alertId\": \"{{alert.id}}\"}";
        }

        TYaChatsType.Builder builder = TYaChatsType.newBuilder()
            .setTextTemplate(tempate);

        if (random.nextBoolean()) {
            builder.setLogin("alexlovkov");
        } else {
            builder.setGroupId("0/0/032dd1f7-f7a9-4c2f-bb03-730068f2c702");
        }

        return builder.build();
    }

    private TDatalensEmailType randomDatalensEmail(ThreadLocalRandom random) {
        var builder = TDatalensEmailType.newBuilder();
        for (int i = random.nextInt(1, 5); i >= 0; i--) {
            builder.addRecipients(UUID.randomUUID().toString());
        }
        return builder.build();
    }
}
