package ru.yandex.solomon.gateway.api;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import io.grpc.Server;
import io.grpc.ServerInterceptor;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.grpc.utils.GrpcServerFactory;
import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.grpc.utils.PublicGrpcService;
import ru.yandex.grpc.utils.RequestParamsInterceptor;
import ru.yandex.monitoring.gateway.YasmGatewayServiceGrpc;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.grpc.GrpcRequestLoggingInterceptor;
import ru.yandex.solomon.conf.db3.ConfigV3DaoContext;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.grpc.GrpcContext;
import ru.yandex.solomon.gateway.api.cloud.priv.v2.QuotaService;
import ru.yandex.solomon.gateway.api.cloud.priv.v2.QuotaServiceImpl;
import ru.yandex.solomon.gateway.api.internal.yasm.YasmGatewayService;
import ru.yandex.solomon.gateway.api.internal.yasm.YasmGatewayServiceImpl;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.ApiV3Context;
import ru.yandex.solomon.gateway.api.v3.cloud.CloudApiV3Context;
import ru.yandex.solomon.gateway.api.v3alpha.cloud.CloudApiV3AlphaContext;
import ru.yandex.solomon.gateway.cloud.api.RequestProducerInterceptor;
import ru.yandex.solomon.gateway.cloud.api.RequestProducerResolver;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.quotas.manager.QuotaManager;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
        GrpcContext.class,
        ConfigDaoContext.class,
        ConfigV3DaoContext.class,
        CloudApiV3Context.class,
        CloudApiV3AlphaContext.class,
        ApiV3Context.class,
})
@SuppressWarnings({"SpringJavaInjectionPointsAutowiringInspection", "OptionalUsedAsFieldOrParameterType"})
@ParametersAreNonnullByDefault
public class ApiGrpcServerContext implements DisposableBean {

    private final GrpcServerFactory grpcServerFactory;

    @WillClose
    @Nullable
    private Server privateApiServer;

    @WillClose
    @Nullable
    private Server publicApiServer;

    @Autowired
    private TGatewayConfig gatewayConfig;

    @Autowired
    public ApiGrpcServerContext(GrpcServerFactory grpcServerFactory) {
        this.grpcServerFactory = grpcServerFactory;
    }

    // Some services.

    @Bean
    QuotaService quotaService(QuotaManager quotaManager) {
        return new QuotaServiceImpl(quotaManager);
    }

    @Bean
    YasmGatewayService yasmGatewayService(DataClient dataClient, AlertingClient alertingClient) {
        return new YasmGatewayServiceImpl(dataClient, alertingClient, gatewayConfig.getYasmGatewayConfig());
    }

    // Interceptors.

    @Bean
    public AuthenticationInterceptor authInterceptor(Authenticator authenticator) {
        return new AuthenticationInterceptor(authenticator, List.of(
                YasmGatewayServiceGrpc.getServiceDescriptor()
        ));
    }

    @Bean
    public RequestProducerInterceptor requestProducerInterceptor(RequestProducerResolver producerResolver) {
        return new RequestProducerInterceptor(producerResolver);
    }

    @Bean
    public Server privateApiMonitoringGrpcServer(
            @Qualifier("GrpcServerConfig") TGrpcServerConfig grpcServerConfig,
            AuthenticationInterceptor authenticationInterceptor,
            RequestProducerInterceptor requestProducerInterceptor,
            @Autowired(required = false) List<GrpcService> grpcServices
    ) {
        List<ServerInterceptor> interceptors = List.of(
                new GrpcRequestLoggingInterceptor(),
                authenticationInterceptor,
                requestProducerInterceptor,
                new EtagInterceptor(),
                new RequestParamsInterceptor());
        this.privateApiServer = this.grpcServerFactory.makePublicServer(
                "GrpcServerConfig",
                grpcServerConfig,
                interceptors,
                grpcServices);
        grpcServerFactory.startWithTimeout(grpcServerConfig, privateApiServer);
        return privateApiServer;
    }

    @Bean
    public Server publicApiMonitoringGrpcServer(
            @Qualifier("PublicGrpcServerConfig") Optional<TGrpcServerConfig> publicGrpcServerConfig,
            AuthenticationInterceptor authenticationInterceptor,
            RequestProducerInterceptor requestProducerInterceptor,
            @Autowired(required = false) List<PublicGrpcService> grpcServices
    ) {
        if (publicGrpcServerConfig.isEmpty()) {
            return null;
        }
        List<ServerInterceptor> interceptors = List.of(
                new GrpcRequestLoggingInterceptor(),
                authenticationInterceptor,
                requestProducerInterceptor,
                new EtagInterceptor(),
                new RequestParamsInterceptor());
        this.publicApiServer = this.grpcServerFactory.makePublicServer(
                "GrpcServerConfig",
                publicGrpcServerConfig.get(),
                interceptors,
                grpcServices);
        grpcServerFactory.startWithTimeout(publicGrpcServerConfig.get(), publicApiServer);
        return publicApiServer;
    }

    @Override
    public void destroy() {
        Server privateServer = this.privateApiServer;
        if (privateServer != null) {
            this.privateApiServer = null;
            privateServer.shutdown();
        }

        Server publicServer = this.publicApiServer;
        if (publicServer != null) {
            this.publicApiServer = null;
            publicServer.shutdown();
        }
    }
}
