package ru.yandex.solomon.gateway.api.cloud.common;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.SelParser;
import ru.yandex.solomon.expression.analytics.ProgramWithReturn;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.AstBinOp;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstInterpolatedString;
import ru.yandex.solomon.expression.ast.AstObject;
import ru.yandex.solomon.expression.ast.AstSelectors;
import ru.yandex.solomon.expression.ast.AstStatement;
import ru.yandex.solomon.expression.ast.AstTernaryOp;
import ru.yandex.solomon.expression.ast.AstUnaryOp;
import ru.yandex.solomon.expression.ast.AstValue;
import ru.yandex.solomon.expression.ast.RecursiveAstVisitor;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ProgramValidator {
    private static ProgramWithReturn parse(String program) {
        return new SelParser(program).parseProgramWithReturn();
    }

    private static void validateNoVars(ProgramWithReturn program) {
        List<AstStatement> statements = program.getStatements();
        if (!statements.isEmpty()) {
            AstStatement first = statements.get(0);
            throw new ProgramValidationException(first.getRange(), "Query cannot contain statements");
        }
    }

    private static class ThrowingAstValidator extends RecursiveAstVisitor<Void, Void> {
        @Override
        protected Void visitIdent(AstIdent ast, Void context) {
            return null;
        }

        @Override
        protected Void visitInterpolatedString(AstInterpolatedString ast, Void context) {
            return null;
        }

        @Override
        protected Void visitBinOp(AstBinOp ast, Void context) {
            return null;
        }

        @Override
        protected Void visitTernaryOp(AstTernaryOp ast, Void context) {
            throw new ProgramValidationException(ast.getRange(), "Query cannot contain ternary operator");
        }

        @Override
        protected Void visitUnaryOp(AstUnaryOp ast, Void context) {
            Ast arg = ast.getParam();
            // Allow +/- for immediate constants
            if (arg instanceof AstValue) {
                return null;
            }
            throw new ProgramValidationException(ast.getRange(), "Query cannot contain unary operators");
        }

        @Override
        protected Void visitCall(AstCall ast, Void context) {
            visit(ast.func, context);
            for (var arg : ast.args) {
                visit(arg, context);
            }
            return null;
        }

        @Override
        protected Void visitValue(AstValue ast, Void context) {
            return null;
        }

        @Override
        protected Void visitSelectors(AstSelectors ast, Void context) {
            return null;
        }

        @Override
        protected Void visitObject(AstObject ast, Void context) {
            throw new ProgramValidationException(ast.getRange(), "Query cannot contain objects");
        }
    }

    public static void validateMonitoringDataApiQuery(String program) {
        ProgramWithReturn parsed = parse(program);

        validateNoVars(parsed);
        new ThrowingAstValidator().visit(parsed.getExpr(), null);
    }
}
