package ru.yandex.solomon.gateway.api.cloud.ext;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ExternalMonitoringMetrics implements MetricSupplier {
    static final ExternalMonitoringMetrics I = new ExternalMonitoringMetrics();

    public static <T> void forFuture(String method, String cloudId, String folderId, CompletableFuture<T> future) {
        I.forFutureImpl(method, cloudId, folderId, future);
    }

    private final ConcurrentMap<String, EndpointMetrics> endpointMetricsById;

    private ExternalMonitoringMetrics() {
        this.endpointMetricsById = new ConcurrentHashMap<>(1000);
    }

    private <T> void forFutureImpl(String method, String cloudId, String folderId, CompletableFuture<T> future) {
        String id = method + "/" + cloudId + "/" + folderId;
        endpointMetricsById.computeIfAbsent(id, id1 -> new EndpointMetrics(method, cloudId, folderId))
                .forFuture(future);
    }

    @Override
    public int estimateCount() {
        return endpointMetricsById.size() * 2;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        for (EndpointMetrics endpointMetrics : endpointMetricsById.values()) {
            endpointMetrics.append(tsMillis, commonLabels, consumer);
        }
    }

    private static class EndpointMetrics implements MetricSupplier {
        private final MetricRegistry registry;
        private final Rate requests;
        private final Rate errors;

        EndpointMetrics(String method, String cloudId, String folderId) {
            Labels labels = Labels.of("method", method, "cloud_id", cloudId, "folder_id", folderId);
            this.registry = new MetricRegistry(labels);
            this.requests = registry.rate("api.http.requests_count_per_second");
            this.errors = registry.rate("api.http.errors_count_per_second");
        }

        public <T> void forFuture(CompletableFuture<T> future) {
            requests.inc();
            future.whenComplete((r, t) -> {
                if (t != null) {
                    errors.inc();
                }
            });
        }

        @Override
        public int estimateCount() {
            return 2;
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
