package ru.yandex.solomon.gateway.api.cloud.priv.v2;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import yandex.cloud.priv.monitoring.v2.QuotaServiceGrpc;
import yandex.cloud.priv.quota.PQ.BatchUpdateQuotaMetricsRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultResponse;
import yandex.cloud.priv.quota.PQ.GetQuotaRequest;
import yandex.cloud.priv.quota.PQ.Quota;
import yandex.cloud.priv.quota.PQ.UpdateQuotaMetricRequest;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Ivan Tsybulin
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcQuotaService extends QuotaServiceGrpc.QuotaServiceImplBase implements GrpcService {
    private final Authorizer authorizer;
    private final QuotaService service;

    @Autowired
    public GrpcQuotaService(Authorizer authorizer, QuotaService service) {
        this.authorizer = authorizer;
        this.service = service;
    }

    @Override
    public void get(GetQuotaRequest request, StreamObserver<Quota> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Quota> getImpl(GetQuotaRequest request) {
        Validators.validate(request);
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        var authFuture =
            authorizer.authorize(authSubject, request.getCloudId(), Permission.QUOTA_GET);
        return authFuture.thenCompose(account -> service.get(request));
    }

    @Override
    public void updateMetric(UpdateQuotaMetricRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> updateMetricImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Empty> updateMetricImpl(UpdateQuotaMetricRequest request) {
        Validators.validate(request);
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        var authFuture =
            authorizer.authorize(authSubject, request.getCloudId(), Permission.QUOTA_UPDATE);
        return authFuture.thenCompose(account -> service.updateMetric(request, account.getId(), Instant.now()))
            .thenApply(unit -> Empty.getDefaultInstance());
    }

    @Override
    public void batchUpdateMetric(BatchUpdateQuotaMetricsRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> batchUpdateMetricImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> batchUpdateMetricImpl(BatchUpdateQuotaMetricsRequest request) {
        Validators.validate(request);
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        var authFuture =
                authorizer.authorize(authSubject, request.getCloudId(), Permission.QUOTA_UPDATE);
        return authFuture.thenCompose(account -> service.batchUpdateMetrics(request, account.getId(), Instant.now()))
            .thenApply(unit -> Empty.getDefaultInstance());
    }

    @Override
    public void getDefault(GetQuotaDefaultRequest request, StreamObserver<GetQuotaDefaultResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getDefaultImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<GetQuotaDefaultResponse> getDefaultImpl(GetQuotaDefaultRequest request) {
        Validators.validate(request);
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        if (authSubject == AnonymousAuthSubject.INSTANCE) {
            throw new StatusRuntimeException(Status.UNAUTHENTICATED.withDescription("Anonymous access is denied"));
        }
        return service.getDefault(request);
    }
}
