package ru.yandex.solomon.gateway.api.cloud.priv.v2;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import yandex.cloud.priv.quota.PQ.BatchUpdateQuotaMetricsRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultResponse;
import yandex.cloud.priv.quota.PQ.GetQuotaRequest;
import yandex.cloud.priv.quota.PQ.MetricLimit;
import yandex.cloud.priv.quota.PQ.Quota;
import yandex.cloud.priv.quota.PQ.QuotaMetric;
import yandex.cloud.priv.quota.PQ.UpdateQuotaMetricRequest;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.quotas.manager.QuotaLimit;
import ru.yandex.solomon.quotas.manager.QuotaManager;
import ru.yandex.solomon.quotas.manager.QuotaValueWithLimit;
import ru.yandex.solomon.quotas.manager.Scope;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class QuotaServiceImpl implements QuotaService {
    private static final String DOMAIN_PREFIX = "monitoring.";

    private final QuotaManager quotaManager;
    private final Collection<String> namespaces;

    public QuotaServiceImpl(QuotaManager quotaManager) {
        this.quotaManager = quotaManager;
        this.namespaces = quotaManager.getRegisteredNamespaces();
    }

    /**
     * Mangling scheme: name = monitoring.<indicator>
     */

    private static String mangle(String indicator) {
        return DOMAIN_PREFIX + indicator;
    }

    private static String demangle(String name) {
        if (!name.startsWith(DOMAIN_PREFIX)) {
            throw new IllegalArgumentException("Quota name is invalid, expecting name starting with '" + DOMAIN_PREFIX +
                    "', but got '" + name + "'");
        }
        return name.substring(DOMAIN_PREFIX.length());
    }

    private static QuotaMetric fromQuotaValueWithLimit(QuotaValueWithLimit quotaValueWithLimit) {
        return QuotaMetric.newBuilder()
            .setName(mangle(quotaValueWithLimit.getIndicator()))
            .setLimit(quotaValueWithLimit.getLimit())
            .setValue(quotaValueWithLimit.getValue())
            .setUsage(quotaValueWithLimit.getValue())
            .build();
    }

    private CompletableFuture<List<QuotaMetric>> getForNamespaceAndMangle(String namespace, String cloudId) {
        Scope scope = Scope.of(namespace, "project", cloudId);
        return quotaManager.getUsageWithLimits(scope)
            .thenApply(quotas -> quotas.stream()
                .map(QuotaServiceImpl::fromQuotaValueWithLimit)
                .collect(Collectors.toList()));
    }

    private static MetricLimit fromQuotaLimit(QuotaLimit quota) {
        return MetricLimit.newBuilder()
            .setName(mangle(quota.getIndicator()))
            .setLimit(quota.getLimit())
            .build();
    }

    private CompletableFuture<List<MetricLimit>> getDefaultsForNamespaceAndMangle(String namespace) {
        return quotaManager.getDefaultLimits(namespace, "project")
            .thenApply(quotas -> quotas.stream()
                .map(QuotaServiceImpl::fromQuotaLimit)
                .collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<Quota> get(GetQuotaRequest request) {
        return namespaces.stream()
            .map(namespace -> getForNamespaceAndMangle(namespace, request.getCloudId()))
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .thenApply(quotaMetricsByNamespace -> {
                Quota.Builder builder = Quota.newBuilder().setCloudId(request.getCloudId());
                for (List<QuotaMetric> quotaMetrics : quotaMetricsByNamespace) {
                    builder.addAllMetrics(quotaMetrics);
                }
                return builder.build();
            });
    }

    private CompletableFuture<Void> updateSingleMetricLimit(
            String cloudId,
            MetricLimit metricLimit,
            String updatedBy,
            Instant updatedAt)
    {
        String indicator = demangle(metricLimit.getName());
        String namespace = quotaManager.getNamespaceByIndicator(indicator)
                .orElseThrow(() -> new IllegalArgumentException("Indicator '" + indicator + "' is unknown"));
        Scope scope = Scope.of(namespace, "project", cloudId);
        return quotaManager.updateLimit(scope, indicator, metricLimit.getLimit(), updatedBy, updatedAt);
    }

    @Override
    public CompletableFuture<Void> updateMetric(UpdateQuotaMetricRequest request, String updatedBy, Instant updatedAt) {
        return updateSingleMetricLimit(request.getCloudId(), request.getMetric(), updatedBy, updatedAt);
    }

    @Override
    public CompletableFuture<Void> batchUpdateMetrics(BatchUpdateQuotaMetricsRequest request, String updatedBy, Instant updatedAt) {
        return request.getMetricsList().stream()
                .map(metricLimit -> updateSingleMetricLimit(request.getCloudId(), metricLimit, updatedBy, updatedAt))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    @Override
    public CompletableFuture<GetQuotaDefaultResponse> getDefault(GetQuotaDefaultRequest request) {
        return namespaces.stream()
            .map(this::getDefaultsForNamespaceAndMangle)
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .thenApply(metricLimitsByNamespace -> {
                GetQuotaDefaultResponse.Builder builder = GetQuotaDefaultResponse.newBuilder();
                for (List<MetricLimit> metricLimits : metricLimitsByNamespace) {
                    builder.addAllMetrics(metricLimits);
                }
                return builder.build();
            });
    }
}
