package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertEvaluationHistoryDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ChannelStatusesHistoryDto;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadRequest;
import ru.yandex.solomon.metrics.client.ReadResponse;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.ResolveOneResponse;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.util.time.Interval;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureMetabaseStatusValid;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureStockpileStatusValid;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingHistoryReader {
    private final MetricsClient metricsClient;
    private final ShardKey alertingStatuses;

    public AlertingHistoryReader(MetricsClient metricsClient, ShardKey alertingStatuses) {
        this.metricsClient = metricsClient;
        this.alertingStatuses = alertingStatuses;
    }

    public CompletableFuture<AlertEvaluationHistoryDto> readEvaluationHistory(
            String cloudId,
            String alertId,
            Interval interval,
            long gridMillis,
            long deadline,
            String subjectId,
            boolean downsamplingOff,
            boolean multiAlert)
    {
        if (multiAlert) {
            return resolveMultiEvaluationMetric(cloudId, alertId, deadline)
                    .thenCompose(keys -> keys.isEmpty() ? completedFuture(List.of()) : readMetrics(keys, interval, gridMillis, deadline, subjectId, null))
                    .thenApply(metrics -> AlertEvaluationHistoryDto.create(metrics, interval, gridMillis));
        }
        return resolveEvaluationMetric(cloudId, alertId, deadline)
            .thenCompose(response -> {
                var status = response.getStatus();
                if (status.getCode() == EMetabaseStatusCode.NOT_FOUND) {
                    return completedFuture(new ReadResponse(response.getMetric(), StockpileStatus.OK));
                }
                ensureMetabaseStatusValid(status);
                return readEvaluationMetric(response.getMetric(), interval, deadline, subjectId, downsamplingOff);
            })
            .thenApply(readResponse -> AlertEvaluationHistoryDto.create(readResponse, interval, gridMillis, downsamplingOff));
    }

    public CompletableFuture<ChannelStatusesHistoryDto> readChannelStatusesHistory(
            String cloudId,
            String channelId,
            Interval interval,
            long gridMillis,
            long deadline,
            String subjectId)
    {
        var downsampling = OperationDownsampling.newBuilder()
                // TODO: Aggregating here will result in possibly smaller diff if COUNTER was reset
                // 1m grid would solve the problem, but looks like an overkill
                // Need something like RATE for integer counters (DIFF?), so aggregation would not loose events
                .setGridMillis(gridMillis)
                .setAggregation(Aggregation.LAST)
                .setFillOption(OperationDownsampling.FillOption.PREVIOUS)
                .build();
        return findChannelStatusMetric(cloudId, channelId, deadline)
            .thenCompose(keys -> keys.isEmpty() ? completedFuture(List.of()) : readMetrics(keys, interval, gridMillis, deadline, subjectId, downsampling))
            .thenApply(ChannelStatusesHistoryDto::create);
    }

    private CompletableFuture<ResolveOneResponse> resolveAlertingMetric(String cloudId, Labels metricLables, long deadline) {
        return metricsClient.resolveOne(ResolveOneRequest.newBuilder()
            .setLabels(Labels.builder(4 + metricLables.size())
                .add(LabelKeys.PROJECT, alertingStatuses.getProject())
                .add(LabelKeys.CLUSTER, alertingStatuses.getCluster())
                .add(LabelKeys.SERVICE, alertingStatuses.getService())
                .add("projectId", cloudId)
                .addAll(metricLables)
                .build())
            .setDeadline(deadline)
            .setProducer(RequestProducer.SYSTEM)
            .build());
    }

    private CompletableFuture<List<MetricKey>> findAlertingMetrics(String cloudId, Selectors metricSelectors, long deadline) {
        return metricsClient.find(FindRequest.newBuilder()
            .setSelectors(Selectors.builder(4 + metricSelectors.size())
                .add(LabelKeys.PROJECT, alertingStatuses.getProject())
                .add(LabelKeys.CLUSTER, alertingStatuses.getCluster())
                .add(LabelKeys.SERVICE, alertingStatuses.getService())
                .add(Selector.exact("projectId", cloudId))
                .addAll(metricSelectors)
                .build())
            .setDeadline(deadline)
            .setProducer(RequestProducer.SYSTEM)
            .build())
            .thenApply(response -> {
                ensureMetabaseStatusValid(response.getStatus());
                return response.getMetrics();
            });
    }

    private CompletableFuture<ResolveOneResponse> resolveEvaluationMetric(String cloudId, String alertId, long deadline) {
        return resolveAlertingMetric(cloudId,
            Labels.of(
                LabelKeys.SENSOR, "alert.evaluation.status",
                "alertId", alertId
            ),
            deadline);
    }

    private CompletableFuture<List<MetricKey>> resolveMultiEvaluationMetric(String cloudId, String alertId, long deadline) {
        return findAlertingMetrics(cloudId,
                Selectors.of(
                        Selector.exact(LabelKeys.SENSOR, "multiAlert.evaluation.status"),
                        Selector.exact("alertId", alertId)
                ),
                deadline);
    }

    private CompletableFuture<List<MetricKey>> findChannelStatusMetric(String cloudId, String channelId, long deadline) {
        return findAlertingMetrics(cloudId,
            Selectors.of(
                Selector.exact(LabelKeys.SENSOR, "channel.notification.status"),
                Selector.exact("channelId", channelId),
                Selector.any("status")
            ),
            deadline);
    }

    private CompletableFuture<ReadResponse> readEvaluationMetric(MetricKey key, Interval interval, long deadline, String subjectId, boolean downsamplingOff) {
        var requestBuilder = ReadRequest.newBuilder()
                .setKey(key)
                .setInterval(interval)
                .setDeadline(deadline)
                .setProducer(RequestProducer.SYSTEM)
                .setSubjectId(subjectId);
        if (!downsamplingOff) {
            // TODO: evaluation interval from alert (@gordiychuk)
            requestBuilder.setGridMillis(TimeUnit.MINUTES.toMillis(1))
                    .setAggregation(Aggregation.LAST);
        }
        return metricsClient.read(requestBuilder.build())
            .thenApply(response -> {
                ensureStockpileStatusValid(response.getStatus());
                return response;
            });
    }

    private CompletableFuture<List<Metric<MetricKey>>> readMetrics(
            List<MetricKey> keys,
            Interval interval,
            long gridMillis,
            long deadline,
            String subjectId,
            OperationDownsampling downsampling)
    {
        // diff needs one interval to the left
        Instant begin = interval.getBegin().minusMillis(gridMillis);
        Instant end = interval.getEnd();
        if (!InstantUtils.isGoodMillis(begin.toEpochMilli())) {
            begin = Instant.ofEpochMilli(InstantUtils.NOT_BEFORE);
        }
        if (!InstantUtils.isGoodMillis(end.toEpochMilli())) {
            end = Instant.ofEpochMilli(InstantUtils.NOT_AFTER);
        }
        Interval extended = new Interval(begin, end);

        var requestBuilder = ReadManyRequest.newBuilder()
                .addKeys(keys)
                .setInterval(downsampling == null
                        ? interval
                        : extended)
                .setDeadline(deadline)
                .setProducer(RequestProducer.SYSTEM)
                .setSubjectId(subjectId);
        if (downsampling != null) {
            requestBuilder.addOperation(Operation.newBuilder()
                    .setDownsampling(downsampling)
                    .build());

        }
        return metricsClient.readMany(requestBuilder.build())
                .thenApply(response -> {
                    ensureStockpileStatusValid(response.getStatus());
                    return response.getMetrics();
                });
    }
}
