package ru.yandex.solomon.gateway.api.cloud.v1;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.notification.TNotificationDetails;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertNotificationStatusDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertState;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertStatusDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertType;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.EvaluationStatusDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.FolderStatsDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ListAlert;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.NotificationChannelPropertiesDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.NotificationChannelPropertiesWithRecipientsDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.OrderDirection;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.cloud.search.NoopSearchEventSink;
import ru.yandex.solomon.gateway.cloud.search.SearchEvent;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSink;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@Api(tags = {"cloud-alerting-alert"})
@RestController
@RequestMapping(path = "/monitoring/v1/alerts", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ConditionalOnBean(TGatewayCloudConfig.class)
@ParametersAreNonnullByDefault
public class CloudAlertController {
    private static final Logger logger = LoggerFactory.getLogger(CloudAlertController.class);

    private final ProjectsDao projectsDao;
    private final AlertingClient alertingClient;
    private final CloudAuthorizer authorizer;
    private final String entityIdPrefix;
    private final SearchEventSink searchEventSink;

    @Autowired
    public CloudAlertController(ProjectsDao projectsDao, AlertingClient alertingClient, CloudAuthorizer authorizer, Optional<SearchEventSink> searchEventSink, Optional<TGatewayCloudConfig> config) {
        this.projectsDao = projectsDao;
        this.alertingClient = alertingClient;
        this.authorizer = authorizer;
        this.searchEventSink = searchEventSink.orElseGet(NoopSearchEventSink::new);
        this.entityIdPrefix = config.map(c -> Strings.emptyToNull(c.getEntityIdPrefix())).orElse(IdGenerator.INTERNAL_PREFIX);
    }

    @ApiOperation(value = "Get alert by id", response = AlertDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/alert", method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<AlertDto> getAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET, cloudId -> doGetAlert(cloudId, folderId, alertId));
    }

    @ApiOperation(value = "Delete alert by id")
    @RequestMapping(path = "/alert", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @SuppressWarnings("unused")
    public CompletableFuture<Void> deleteAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_DELETE, cloudId -> doDeleteAlert(cloudId, folderId, alertId));
    }

    @ApiOperation(value = "Create alert", response = AlertDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/alert", method = RequestMethod.POST)
    @SuppressWarnings("unused")
    public CompletableFuture<AlertDto> createAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestBody AlertDto alert)
    {
        alert.fillCreatedNow(folderId, subject.getUniqueId());
        alert.id = IdGenerator.generateId(entityIdPrefix);

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_CREATE, cloudId -> doCreateAlert(cloudId, folderId, alert));
    }

    @ApiOperation(value = "Update alert", response = AlertDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/alert", method = RequestMethod.PUT)
    @SuppressWarnings("unused")
    public CompletableFuture<AlertDto> updateAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId,
        @RequestBody AlertDto alert)
    {
        alert.fillUpdatedNow(folderId, alertId, subject.getUniqueId());

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_UPDATE, cloudId -> doUpdateAlert(cloudId, folderId, alert));
    }

    @ApiOperation(value = "List alerts", response = ListAlert.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<ListAlert> listAlerts(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
        @RequestParam(value = "filterByStates", required = false) @Nullable List<AlertState> filterByState,
        @RequestParam(value = "filterByTypes", required = false) @Nullable List<AlertType> filterByType,
        @RequestParam(value = "filterByEvaluationStatus", required = false) @Nullable List<EvaluationStatusDto.Code> filterByEvaluation,
        @RequestParam(value = "filterByNotificationId", required = false) @Nullable List<String> filterByNotificationId,
        @RequestParam(value = "orderByName", required = false) @Nullable OrderDirection orderByName,
        @RequestParam(value = "orderByState", required = false) @Nullable OrderDirection orderByState,
        @RequestParam(value = "orderByType", required = false) @Nullable OrderDirection orderByType,
        @RequestParam(value = "pageSize", defaultValue = "30", required = false) int pageSize,
        @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_LIST, cloudId ->
            doListAlerts(cloudId, folderId,
                filterByName, filterByState, filterByType, filterByEvaluation, filterByNotificationId,
                orderByName, orderByState, orderByType,
                pageSize, pageToken));
    }

    @ApiOperation(value = "Get alert evaluation state", response = AlertStatusDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/state/evaluation", method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<AlertStatusDto> readEvaluationState(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET, cloudId -> doReadEvaluationState(cloudId, folderId, alertId));
    }

    @ApiOperation(value = "Get alert notification state", response = AlertNotificationStatusDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/state/notification", method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<AlertNotificationStatusDto> readNotificationState(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET, cloudId -> doReadNotificationState(cloudId, folderId, alertId));
    }

    @ApiOperation(value = "Get folder stats", response = FolderStatsDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/stats", method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<FolderStatsDto> getFolderStats(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_LIST, cloudId -> doFolderStats(cloudId, folderId));
    }

    private CompletableFuture<AlertDto> doGetAlert(String cloudId, String folderId, String alertId) {
        return readAlert(cloudId, folderId, alertId)
                .thenCompose(alert -> resolveChannelsForAlert(alert, cloudId, folderId));
    }

    private CompletableFuture<AlertDto> resolveChannelsForAlert(TAlert alert, String cloudId, String folderId) {
        Set<String> channelIds = alert.getConfiguredNotificationChannelsMap().keySet();

        return alertingClient.resolveNotificationDetails(TResolveNotificationDetailsRequest.newBuilder()
            .setProjectId(cloudId)
            .setFolderId(folderId)
            .addAllNotificationIds(channelIds)
            .build())
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                List<NotificationChannelPropertiesWithRecipientsDto> channelProps = response.getNotificationDetailsList().stream()
                    .map(NotificationChannelPropertiesWithRecipientsDto::fromProto)
                    .collect(Collectors.toList());
                return AlertDto.fromProto(alert, channelProps);
            });
    }

    private CompletableFuture<AlertDto> doCreateAlert(String cloudId, String folderId, AlertDto alert) {
        return checkProjectExistence(cloudId)
            .thenCompose(aVoid -> createAlert(alert.toProto(cloudId)))
            .thenCompose(created -> resolveChannelsForAlert(created, cloudId, folderId));
    }

    private CompletableFuture<AlertDto> doUpdateAlert(String cloudId, String folderId, AlertDto alert) {
        return checkProjectExistence(cloudId)
            .thenCompose(ignore -> updateAlert(alert.toProto(cloudId)))
            .thenCompose(updated -> resolveChannelsForAlert(updated, cloudId, folderId));
    }

    private CompletableFuture<ListAlert> doListAlerts(
            String cloudId,
            String folderId,
            String filterByName,
            @Nullable List<AlertState> filterByState,
            @Nullable List<AlertType> filterByType,
            @Nullable List<EvaluationStatusDto.Code> filterByEvaluation,
            @Nullable List<String> filterByNotificationId,
            @Nullable OrderDirection orderByName,
            @Nullable OrderDirection orderByState,
            @Nullable OrderDirection orderByType,
            int pageSize,
            String pageToken)
    {
        return alertingClient.listAlerts(TListAlertRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setFilterByName(filterByName)
                .addAllFilterByState(Nullables.orEmpty(filterByState).stream().map(AlertState::toProto).collect(Collectors.toList()))
                .addAllFilterByType(Nullables.orEmpty(filterByType).stream().map(AlertType::toProto).collect(Collectors.toList()))
                .addAllFilterByEvaluationStatus(Nullables.orEmpty(filterByEvaluation).stream().map(EvaluationStatusDto.Code::toProto).collect(Collectors.toList()))
                .addAllFilterByNotificationId(Nullables.orEmpty(filterByNotificationId))
                .setOrderByName(Nullables.orDefault(orderByName, OrderDirection.ASC).toProto())
                .setOrderByState(Nullables.orDefault(orderByState, OrderDirection.ASC).toProto())
                .setOrderByType(Nullables.orDefault(orderByType, OrderDirection.ASC).toProto())
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build()
            )
            .thenCompose(listAlertResponse -> {
                ensureStatusValid(listAlertResponse.getRequestStatus(), listAlertResponse::getStatusMessage);
                var channelIds = listAlertResponse.getAlertsList().stream()
                    .flatMap(alert -> alert.getNotificationChannelIdsList().stream())
                    .collect(Collectors.toSet());
                return alertingClient.resolveNotificationDetails(TResolveNotificationDetailsRequest.newBuilder()
                        .setProjectId(cloudId)
                        .setFolderId(folderId)
                        .addAllNotificationIds(channelIds)
                        .build())
                    .thenApply(response -> {
                        ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                        var notificationDetailsById = response.getNotificationDetailsList().stream()
                            .collect(Collectors.toMap(TNotificationDetails::getNotificationId, Function.identity()));
                        return ListAlert.fromProto(
                            listAlertResponse,
                            (id, dto) -> {
                                TNotificationDetails details = notificationDetailsById.get(id);
                                if (details == null) {
                                    NotificationChannelPropertiesDto.fillNotFound(dto, id);
                                } else {
                                    NotificationChannelPropertiesDto.fillFromProto(dto, details);
                                }
                            });
                    });
            });
    }

    private CompletableFuture<Void> doDeleteAlert(String cloudId, String folderId, String alertId) {
        return readAlert(cloudId, folderId, alertId)
                .thenCompose(this::deleteAlert);
    }

    private CompletableFuture<TAlert> readAlert(String cloudId, String folderId, String alertId) {
        var req = TReadAlertRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setAlertId(alertId)
                .build();

        return alertingClient.readAlert(req)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return response.getAlert();
                });
    }

    private CompletableFuture<Void> deleteAlert(TAlert alert) {
        long now = System.currentTimeMillis();
        var req = TDeleteAlertRequest.newBuilder()
                .setProjectId(alert.getProjectId())
                .setFolderId(alert.getFolderId())
                .setAlertId(alert.getId())
                .build();

        return alertingClient.deleteAlert(req)
                .thenAccept(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    searchEventSink.accept(SearchEvent.alert(alert).deletedAt(now));
                });
    }

    private CompletableFuture<TAlert> createAlert(TAlert alert) {
        var req = TCreateAlertRequest.newBuilder()
                .setAlert(alert)
                .build();

        return alertingClient.createAlert(req)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    var created = response.getAlert();
                    searchEventSink.accept(SearchEvent.alert(created));
                    return created;
                });
    }

    private CompletableFuture<TAlert> updateAlert(TAlert alert) {
        var req = TUpdateAlertRequest.newBuilder()
                .setAlert(alert)
                .build();

        return alertingClient.updateAlert(req)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    var updated = response.getAlert();
                    searchEventSink.accept(SearchEvent.alert(updated));
                    return updated;
                });
    }

    private CompletableFuture<AlertStatusDto> doReadEvaluationState(String cloudId, String folderId, String alertId) {
        return alertingClient.readEvaluationState(TReadEvaluationStateRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setAlertId(alertId)
                .build())
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertStatusDto.fromProto(response.getState());
            });
    }

    private CompletableFuture<AlertNotificationStatusDto> doReadNotificationState(String cloudId, String folderId, String alertId) {
        return alertingClient.readNotificationState(TReadNotificationStateRequest.newBuilder()
            .setProjectId(cloudId)
            .setFolderId(folderId)
            .setAlertId(alertId)
            .build())
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return AlertNotificationStatusDto.fromProto(response.getStatesList());
            });
    }

    private CompletableFuture<FolderStatsDto> doFolderStats(String cloudId, String folderId) {
        TReadProjectStatsRequest request = TReadProjectStatsRequest.newBuilder()
            .setProjectId(cloudId)
            .setFolderId(folderId)
            .build();
        return alertingClient.readProjectStats(request)
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return FolderStatsDto.fromProto(response);
            });
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        RequestStatusToAlertingException.throwIfNotOk(statusCode, messageFn);
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("Alerting project %s does not exist", projectId));
                }
            });
    }
}
