package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertEvaluationHistoryDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@Api(tags = {"cloud-alerting-evaluation"})
@RestController
@RequestMapping(path = "/monitoring/v1/alerts/evaluation", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class CloudAlertEvaluationController {
    private final CloudAuthorizer authorizer;
    private final AlertingHistoryReader historyReader;
    private final AlertApi alertApi;

    @Autowired
    public CloudAlertEvaluationController(
            CloudAuthorizer authorizer,
            MetricsClient metricsClient,
            @Qualifier("alertingStatuses") ShardKey alertingStatuses,
            AlertApi alertApi)
    {
        this.authorizer = authorizer;
        this.alertApi = alertApi;
        this.historyReader = new AlertingHistoryReader(metricsClient, alertingStatuses);
    }

    @ApiOperation(value = "Get alert evaluation history", response = AlertEvaluationHistoryDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/history", method = RequestMethod.GET)
    public CompletableFuture<AlertEvaluationHistoryDto> readEvaluationHistory(
            @RequireAuth AuthSubject subject,
            @RequestParam("folderId") String folderId,
            @RequestParam("alertId") String alertId,
            @RequestParam("from") Instant from,
            @RequestParam("to") Instant to,
            @RequestParam(value = "gridMillis", defaultValue = "60000") long gridMillis,
            @RequestParam(value = "downsamplingOff", defaultValue = "false") boolean downsamplingOff) {
        ValidationUtils.validateInterval(from, to);
        var interval = Interval.extendToGrid(new Interval(from, to), gridMillis);
        long deadline = System.currentTimeMillis() + Deadline.DEFAULT_TIMEOUT_MILLIS;
        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET,
                cloudId -> alertApi.readAlert(TReadAlertInterpolatedRequest.newBuilder()
                        .setAlertId(alertId)
                        .setProjectId(cloudId)
                        .build())
                        .thenCompose(response -> {
                            var multi = response.getInterpolatedAlert().getGroupByLabelsCount() > 0;
                            return historyReader.readEvaluationHistory(cloudId, alertId, interval, gridMillis, deadline, subjectId, downsamplingOff, multi);
                        }));
    }
}
