package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AlertSimulationDto;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.util.time.Deadline;

/**
 * @author Ivan Tsybulin
 */
@Api(tags = {"cloud-alerting-simulation"})
@RestController
@RequestMapping(path = "/monitoring/v1/alerts/simulation", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class CloudAlertSimulatorController {
    private final CloudAuthorizer authorizer;

    private final AlertingClient alertingClient;

    @Autowired
    public CloudAlertSimulatorController(CloudAuthorizer authorizer, AlertingClient alertingClient) {
        this.authorizer = authorizer;
        this.alertingClient = alertingClient;
    }

    @ApiOperation(value = "Simulate existing alert behavior", response = AlertSimulationDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/alert", method = RequestMethod.GET)
    public CompletableFuture<AlertSimulationDto> simulateExistingAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("alertId") String alertId,
        @RequestParam("from") Instant from,
        @RequestParam("to") Instant to,
        @RequestParam(value = "gridMillis", defaultValue = "60000") long gridMillis)
    {
        ValidationUtils.validateInterval(from, to);
        validatePoints(from, to, gridMillis);
        long deadline = System.currentTimeMillis() + Deadline.DEFAULT_TIMEOUT_MILLIS;
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET, cloudId ->
            simulate(TSimulateEvaluationRequest.newBuilder()
                .setReference(TExplainEvaluationRequest.TReferenceToAlert.newBuilder()
                    .setProjectId(cloudId)
                    .setFolderId(folderId)
                    .setAlertId(alertId))
                .setGridMillis(gridMillis)
                .setSimulationTimeBeginMillis(from.toEpochMilli())
                .setSimulationTimeEndMillis(to.toEpochMilli())
                .setDeadlineMillis(deadline)
                .build())
            .thenApply(simulateResponse -> {
                RequestStatusToAlertingException.throwIfNotOk(simulateResponse.getRequestStatus(), simulateResponse::getStatusMessage);
                return AlertSimulationDto.fromProto(simulateResponse);
            }));
    }

    @ApiOperation(value = "Simulate new alert behavior", response = AlertSimulationDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/alert", method = RequestMethod.POST)
    public CompletableFuture<AlertSimulationDto> simulateNewAlert(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("from") Instant from,
        @RequestParam("to") Instant to,
        @RequestParam(value = "gridMillis", defaultValue = "60000") long gridMillis,
        @RequestBody AlertDto alert)
    {
        ValidationUtils.validateInterval(from, to);
        validatePoints(from, to, gridMillis);
        alert.fillCreatedNow(folderId, subject.getUniqueId());
        long deadline = System.currentTimeMillis() + Deadline.DEFAULT_TIMEOUT_MILLIS;

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.DATA_READ, cloudId ->
            simulate(TSimulateEvaluationRequest.newBuilder()
                .setAlert(alert.toProto(cloudId))
                .setGridMillis(gridMillis)
                .setSimulationTimeBeginMillis(from.toEpochMilli())
                .setSimulationTimeEndMillis(to.toEpochMilli())
                .setDeadlineMillis(deadline)
                .build())
            .thenApply(simulateResponse -> {
                RequestStatusToAlertingException.throwIfNotOk(simulateResponse.getRequestStatus(), simulateResponse::getStatusMessage);
                return AlertSimulationDto.fromProto(simulateResponse);
            }));
    }

    private static void validatePoints(Instant from, Instant to, long gridMillis) {
        long maxPoints = Duration.ofDays(2).toMinutes();
        if (Duration.between(from, to).toMillis() > maxPoints * gridMillis) {
            throw new BadRequestException("Too many points requested, the limit is " + maxPoints + ". Please increase gridMillis");
        }
    }

    private CompletableFuture<TSimulateEvaluationResponse> simulate(TSimulateEvaluationRequest request) {
        return alertingClient.simulateEvaluation(request);
    }
}
