package ru.yandex.solomon.gateway.api.cloud.v1;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class CloudAuthorizer {

    /**
     * These are ids of specially configured accounts in the Cloud.
     * Only these accounts are permitted to create shards with arbitrary services.
     */
    private static final Set<String> GOD_ACCOUNTS = Set.of("aje42000000000000016", "bfb40000000000000016");

    private Authorizer authorizer;

    // Cloud by folder resolved isn't implemented for not cloud Solomon instances
    @Nullable
    private CloudByFolderResolver cloudByFolderResolver;

    @Autowired
    public CloudAuthorizer(Authorizer authorizer, Optional<CloudByFolderResolver> cloudByFolderResolver) {
        this.authorizer = authorizer;
        this.cloudByFolderResolver = cloudByFolderResolver.orElse(null);
    }

    public static void validateFolderId(String folderId) {
        if (!StrictValidator.SELF.isValueValid(folderId)) {
            throw new BadRequestException("invalid value of folder_id: \'" + folderId + '\'');
        }
    }

    private CompletableFuture<Pair<Account, String>> authorizeAndResolveCloudId(
        AuthSubject subject,
        String folderId,
        Permission permission)
    {
        validateFolderId(folderId);

        // projectId/cloudId is not required in cloud authorization
        return authorizer.authorize(subject, "", folderId, permission)
            .thenCompose(account -> {
                if (cloudByFolderResolver == null) {
                    throw new IllegalStateException("cannot resolve cloud id by folder id in not-cloud gateway");
                }

                return cloudByFolderResolver.resolveCloudId(folderId)
                    .thenApply(cloudId -> Pair.of(account, cloudId));
            });
    }

    public <T> CompletableFuture<T> authorizeAndResolveCloudId(
        AuthSubject subject,
        String folderId,
        Permission permission,
        BiFunction<Account, String, CompletableFuture<T>> future)
    {
        return authorizeAndResolveCloudId(subject, folderId, permission)
            .thenCompose(pair -> future.apply(pair.getLeft(), pair.getRight()));
    }

    public  <T> CompletableFuture<T> authorizeAndResolveCloudId(
        AuthSubject subject,
        String folderId,
        Permission permission,
        Function<String, CompletableFuture<T>> future)
    {
        return authorizeAndResolveCloudId(subject, folderId, permission)
            .thenCompose(pair -> future.apply(pair.getRight()));
    }

    public boolean isGodUser(String accountId) {
        return GOD_ACCOUNTS.contains(accountId);
    }
}
