package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.api.cloud.RequestProducerContext;
import ru.yandex.solomon.gateway.api.cloud.common.ProgramValidator;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataRequestDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataResultDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.DownsamplingDto;
import ru.yandex.solomon.gateway.cloud.api.RequestProducerResolver;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling.FillOption;
import ru.yandex.solomon.util.time.Deadline;


/**
 * @author Sergey Polovko
 */
@Import(RequestProducerContext.class)
@RestController
@RequestMapping(path = "/monitoring/v1/data/read", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@SuppressWarnings("unused")
public class CloudDataReadController {

    private final DataClient dataClient;
    private final CloudAuthorizer authorizer;
    private final RequestProducerResolver producerResolver;

    @Autowired
    public CloudDataReadController(
            DataClient dataClient,
            CloudAuthorizer authorizer,
            RequestProducerResolver producerResolver)
    {
        this.dataClient = dataClient;
        this.authorizer = authorizer;
        this.producerResolver = producerResolver;
    }

    @RequestMapping(method = RequestMethod.POST, consumes = MediaType.TEXT_PLAIN_VALUE)
    CompletableFuture<DataResultDto> readDataFromText(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam("from") Instant from,
        @RequestParam("to") Instant to,
        @RequestParam(value = "maxPoints", defaultValue = "0") int maxPointsParam,
        @RequestParam(value = "gridMillis", defaultValue = "0") long gridMillis,
        @RequestParam(value = "downsamplingAggregation", defaultValue = "DEFAULT_AGGREGATION") Aggregation downsamplingAggr,
        @RequestParam(value = "downsamplingFill", defaultValue = "NULL") FillOption downsamplingFill,
        @RequestBody String query,
        @RequestHeader(name = "x-ui-request", required = false, defaultValue = "") String uiHeader)
    {
        DataRequestDto dto = new DataRequestDto();

        dto.program = query;
        dto.from = from;
        dto.to = to;
        dto.downsampling = new DownsamplingDto();
        dto.downsampling.maxPoints = maxPointsParam;
        dto.downsampling.gridMillis = gridMillis;
        dto.downsampling.aggregation = downsamplingAggr;
        dto.downsampling.fill = downsamplingFill;

        return readDataFromJson(subject, folderId, dto, uiHeader);
    }

    @RequestMapping(method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    CompletableFuture<DataResultDto> readDataFromJson(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestBody DataRequestDto requestDto,
        @RequestHeader(name = "x-ui-request", required = false, defaultValue = "") String uiHeader)
    {
        Instant deadline = Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
        requestDto.useNewFormat = true;
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.DATA_READ, cloudId -> {
            if (requestDto.downsampling != null && requestDto.downsampling.points != null) {
                return CompletableFuture.failedFuture(new BadRequestException("\"points\" parameter isn't supported"));
            }

            requestDto.forceCluster = "";
            requestDto.validate();

            ProgramValidator.validateMonitoringDataApiQuery(requestDto.program);

            RequestProducer producer = producerResolver.resolve(uiHeader);
            String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());
            DataRequest request = requestDto.toModel(cloudId, folderId, producer, deadline, subjectId).toBuilder()
                    .setVersion(SelVersion.GROUP_LINES_RETURN_VECTOR_2)
                    .build();
            return dataClient.readData(request)
                .thenApply(DataResultDto::fromModel);
        });
    }
}
