package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.AllLabelValuesResponseDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.LabelKeysResponseDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.LabelValuesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.MetricDto;
import ru.yandex.solomon.gateway.api.v2.dto.MetricNamesResponseDto;
import ru.yandex.solomon.gateway.api.v2.dto.PagedResultDto;
import ru.yandex.solomon.gateway.api.v2.managers.MetricsManager;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.ydb.page.PageOptions;


/**
 * @author Sergey Polovko
 */
@RestController
@RequestMapping(path = "/monitoring/v1/metrics", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ MetricsManager.class})
@SuppressWarnings("unused")
public class CloudMetricsController {

    private static final LabelValidationFilter validationFilter = LabelValidationFilter.ALL;

    @Autowired
    private CloudAuthorizer authorizer;

    @Autowired
    private MetricsManager metricsManager;

    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<PagedResultDto<MetricDto>> findMetrics(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        PageOptions pageOptions)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRICS_GET, cloudId -> {
            return findMetricsImpl(cloudId, folderId, selectorsStr, pageOptions);
        });
    }

    private CompletableFuture<PagedResultDto<MetricDto>> findMetricsImpl(
        String cloudId,
        String folderId,
        String selectorsStr,
        PageOptions pageOptions)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager.findMetrics(selectors, "", pageOptions, true)
            .thenApply(metrics -> PagedResultDto.fromModel(metrics, MetricDto::fromModel));
    }

    @RequestMapping(path = "/names", method = RequestMethod.GET)
    CompletableFuture<MetricNamesResponseDto> findMetricNames(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "limit", defaultValue = "0") int limit)
    {
        Instant deadline = deadline();

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_LABELS_GET, cloudId -> {
            return findMetricNamesImpl(cloudId, folderId, selectorsStr, text, limit, deadline);
        });
    }

    private CompletableFuture<MetricNamesResponseDto> findMetricNamesImpl(
        String cloudId,
        String folderId,
        String selectorsStr,
        String text,
        int limit,
        Instant deadline)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);

        return metricsManager.findMetricNames(selectors, text, LabelValidationFilter.ALL, limit, "", deadline)
            .thenApply(MetricNamesResponseDto::fromModel);
    }

    @RequestMapping(path = "/labelValues", method = RequestMethod.GET)
    CompletableFuture<AllLabelValuesResponseDto> findAllLabelValuesInNewFormat(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "limit", defaultValue = "1000") int limit)
    {
        Instant deadline = deadline();

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_LABELS_GET, cloudId -> {
            return findAllLabelValuesImpl(cloudId, folderId, selectorsStr, text, limit, deadline);
        });
    }

    private CompletableFuture<AllLabelValuesResponseDto> findAllLabelValuesImpl(
        String cloudId,
        String folderId,
        String selectorsStr,
        String text,
        int limit,
        Instant deadline)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);

        return metricsManager.findLabelValues(
            Collections.emptyList(),
            selectors,
            text,
            validationFilter,
            limit,
            true,
            "",
            deadline)
            .thenApply(
                model -> AllLabelValuesResponseDto.fromModel(model, selectors, validationFilter));
    }

    @RequestMapping(path = "/labelKeys", method = RequestMethod.GET)
    CompletableFuture<LabelKeysResponseDto> findLabelKeysInNewFormat(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr)
    {
        Instant deadline = deadline();

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_NAMES_GET, cloudId -> {
            return findLabelKeysImpl(cloudId, folderId, selectorsStr, deadline);
        });
    }

    private CompletableFuture<LabelKeysResponseDto> findLabelKeysImpl(
        String cloudId,
        String folderId,
        String selectorsStr,
        Instant deadline)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager.findLabelNames(selectors, true, "", deadline)
            .thenApply(LabelKeysResponseDto::fromModel);
    }

    @RequestMapping(path = "/labelKeys/values", method = RequestMethod.GET)
    CompletableFuture<LabelValuesResponseDto> findAllLabelValuesInNewFormat(
        @RequireAuth AuthSubject subject,
        @RequestParam("folder_id") String folderId,
        @RequestParam("labelKey") String labelKey,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "limit", defaultValue = "1000") int limit)
    {
        Instant deadline = deadline();

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_LABELS_GET, cloudId -> {
            return findAllLabelValuesImpl(cloudId, folderId, labelKey, selectorsStr, text, limit, deadline);
        });
    }

    private CompletableFuture<LabelValuesResponseDto> findAllLabelValuesImpl(
        String cloudId,
        String folderId,
        String labelKey,
        String selectorsStr,
        String text,
        int limit,
        Instant deadline)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);

        return metricsManager
            .findLabelValues(List.of(labelKey), selectors, text, validationFilter, limit, true, "", deadline)
            .thenApply(model -> LabelValuesResponseDto.fromModel(model, validationFilter));
    }

    private Selectors parseSelectors(String cloudId, String folderId, String selectors) {
        try {
            return Selectors.parse(selectors)
                .toBuilder()
                .addOverride(Selector.exact(LabelKeys.PROJECT, cloudId))
                .addOverride(Selector.exact(LabelKeys.CLUSTER, folderId))
                .build();
        } catch (Throwable e) {
            throw new BadRequestException(String.format("failed to parse selectors %s", selectors));
        }
    }

    private static Instant deadline() {
        return Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
    }
}
