package ru.yandex.solomon.gateway.api.cloud.v1;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.ListNotification;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.NotificationChannelDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.OrderDirection;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.TransportType;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@Api(tags = {"cloud-alerting-notification-channel"})
@RestController
@RequestMapping(path = "/monitoring/v1/channels", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class CloudNotificationChannelController {
    private static final Logger logger = LoggerFactory.getLogger(CloudNotificationChannelController.class);

    private final ProjectsDao projectsDao;
    private final AlertingClient alertingClient;
    private final CloudAuthorizer authorizer;
    private final String entityIdPrefix;

    @Autowired
    public CloudNotificationChannelController(ProjectsDao projectsDao, AlertingClient alertingClient, CloudAuthorizer authorizer, Optional<TGatewayCloudConfig> config) {
        this.projectsDao = projectsDao;
        this.alertingClient = alertingClient;
        this.authorizer = authorizer;
        this.entityIdPrefix = config.map(c -> Strings.emptyToNull(c.getEntityIdPrefix())).orElse(IdGenerator.INTERNAL_PREFIX);
    }

    @ApiOperation(value = "Get notification channel by id", response = NotificationChannelDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/channel", method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<NotificationChannelDto> getNotification(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("notificationId") String notificationId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_GET, cloudId -> doGetNotificationChannel(cloudId, folderId, notificationId));
    }

    @ApiOperation(value = "Delete notification channel by id")
    @RequestMapping(path = "/channel", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @SuppressWarnings("unused")
    public CompletableFuture<Void> deleteNotification(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("notificationId") String notificationId)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_DELETE, cloudId -> doDeleteNotification(cloudId, folderId, notificationId));
    }

    @ApiOperation(value = "Create notification channel", response = NotificationChannelDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/channel", method = RequestMethod.POST)
    @SuppressWarnings("unused")
    public CompletableFuture<NotificationChannelDto> createNotification(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestBody NotificationChannelDto notificationChannel)
    {
        notificationChannel.fillCreatedNow(folderId, subject.getUniqueId());
        notificationChannel.id = IdGenerator.generateId(entityIdPrefix);

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_CREATE, cloudId -> doCreateNotification(cloudId, notificationChannel));
    }

    @ApiOperation(value = "Update notification channel", response = NotificationChannelDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/channel", method = RequestMethod.PUT)
    @SuppressWarnings("unused")
    public CompletableFuture<NotificationChannelDto> updateNotification(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("notificationId") String notificationId,
        @RequestBody NotificationChannelDto notificationChannel)
    {
        notificationChannel.fillUpdatedNow(folderId, notificationId, subject.getUniqueId());

        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_UPDATE, cloudId -> doUpdateNotification(cloudId, notificationChannel));
    }

    @ApiOperation(value = "List notification channels", response = ListNotification.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(method = RequestMethod.GET)
    @SuppressWarnings("unused")
    public CompletableFuture<ListNotification> listNotifications(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
        @RequestParam(value = "filterByType", required = false) @Nullable List<TransportType> filterByType,
        @RequestParam(value = "orderByName", required = false) @Nullable OrderDirection orderByName,
        @RequestParam(value = "orderByType", required = false) @Nullable OrderDirection orderByType,
        @RequestParam(value = "pageSize", defaultValue = "30", required = false) int pageSize,
        @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.CONFIGS_LIST, cloudId ->
            doListNotifications(cloudId, folderId, filterByName, filterByType, orderByName, orderByType, pageSize, pageToken));
    }

    private CompletableFuture<NotificationChannelDto> doGetNotificationChannel(String cloudId, String folderId, String notificationId) {
        return alertingClient
            .readNotification(TReadNotificationRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setNotificationId(notificationId)
                .build()
            )
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                TNotification notification = response.getNotification();
                return NotificationChannelDto.fromProto(notification);
            });
    }

    private CompletableFuture<NotificationChannelDto> doCreateNotification(String cloudId, NotificationChannelDto notificationChannel) {
        return checkProjectExistence(cloudId)
            .thenCompose(aVoid -> alertingClient.createNotification(TCreateNotificationRequest.newBuilder()
                .setNotification(notificationChannel.toProto(cloudId))
                .build()))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelDto.fromProto(response.getNotification());
            });
    }

    private CompletableFuture<NotificationChannelDto> doUpdateNotification(String cloudId, NotificationChannelDto notificationChannel) {
        return checkProjectExistence(cloudId)
            .thenCompose(aVoid -> alertingClient.updateNotification(TUpdateNotificationRequest.newBuilder()
                .setNotification(notificationChannel.toProto(cloudId))
                .build()))
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return NotificationChannelDto.fromProto(response.getNotification());
            });
    }

    private CompletableFuture<ListNotification> doListNotifications(String cloudId, String folderId, String filterByName,
                                                                    @Nullable List<TransportType> filterByType,
                                                                    @Nullable OrderDirection orderByName, @Nullable OrderDirection orderByType,
                                                                    int pageSize, String pageToken) {
        return alertingClient.listNotification(TListNotificationsRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setFilterByName(filterByName)
                .addAllFilterByType(Nullables.orEmpty(filterByType).stream().map(TransportType::toProto).collect(Collectors.toList()))
                .setOrderByName(Nullables.orDefault(orderByName, OrderDirection.ASC).toProto())
                .setOrderByType(Nullables.orDefault(orderByType, OrderDirection.ASC).toProto())
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build()
            )
            .thenApply(response -> {
                ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                return ListNotification.fromProto(response);
            });
    }

    private CompletableFuture<Void> doDeleteNotification(String cloudId, String folderId, String notificationId) {
        return alertingClient.deleteNotification(TDeleteNotificationRequest.newBuilder()
                .setProjectId(cloudId)
                .setFolderId(folderId)
                .setNotificationId(notificationId)
                .build())
            .thenAccept(response -> ensureStatusValid(response.getRequestStatus(), response::getStatusMessage));
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        RequestStatusToAlertingException.throwIfNotOk(statusCode, messageFn);
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("Alerting project %s does not exist", projectId));
                }
            });
    }
}
