package ru.yandex.solomon.gateway.api.cloud.v1;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.expression.SelParser;
import ru.yandex.solomon.expression.analytics.ProgramWithReturn;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.serialization.AstMappingContext;
import ru.yandex.solomon.expression.exceptions.ParserException;
import ru.yandex.solomon.expression.expr.SelFunctions;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.FuncDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ParseErrorDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ParseRequestDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ParseResponseDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ProgramAstDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.RenderErrorDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.RenderRequestDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.RenderResponseDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.TypeDto;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@Api(tags = {"expression"})
@RestController
@RequestMapping(path = "/monitoring/v1/expression", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@ParametersAreNonnullByDefault
public class ExpressionController {

    @ApiOperation(value = "List function", response = FuncDto.class, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @RequestMapping(path = "/functions", method = RequestMethod.GET)
    public List<FuncDto> listFunctions(
        @RequireAuth AuthSubject subject,
        @Nullable @RequestParam(value = "input", required = false) TypeDto input,
        @Nullable @RequestParam(value = "output", required = false) TypeDto output,
        @Nullable @RequestParam(value = "category", required = false) SelFuncCategory category,
        @Nullable @RequestParam(value = "version", required = false) SelVersion version)
    {
        return SelFunctions.REGISTRY.stream(Nullables.orDefault(version, SelVersion.CURRENT))
            .sorted(Comparator.comparing(SelFunc::getName))
            .filter(func -> category == null || category == func.getCategory())
            .map(FuncDto::toDto)
            .filter(Objects::nonNull)
            .filter(func -> output == null || func.returnType == output)
            .filter(func -> input == null || func.args.stream().anyMatch(arg -> arg.type == input))
            .collect(Collectors.toList());
    }

    @ApiOperation(
            value = "Parse expression into ast tree",
            response = ParseResponseDto.class,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @RequestMapping(path = "/parse", method = RequestMethod.POST)
    public ParseResponseDto parseExpression(
            @RequireAuth AuthSubject subject,
            @RequestBody ParseRequestDto request)
    {
        ParseResponseDto result = new ParseResponseDto();
        String source = Nullables.orEmpty(request.program);
        boolean posInfo = Nullables.orFalse(request.needPositions);
        try {
            ProgramWithReturn programWithReturn = new SelParser(source, true).parseProgramWithReturn();
            result.programAst = new ProgramAstDto(programWithReturn, posInfo);
        } catch (ParserException e) {
            result.firstParsingError = new ParseErrorDto(e);
        }
        return result;
    }

    @ApiOperation(
            value = "Render expression text from ast tree",
            response = RenderResponseDto.class,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @RequestMapping(path = "/render", method = RequestMethod.POST)
    public RenderResponseDto renderExpression(
            @RequireAuth AuthSubject subject,
            @RequestBody RenderRequestDto request)
    {
        RenderResponseDto result = new RenderResponseDto();
        try {
            AstMappingContext mapper = new AstMappingContext(false);
            Ast ast = mapper.parse(request.ast);
            result.program = mapper.renderToString(ast);
        } catch (RuntimeException e) {
            result.renderingError = new RenderErrorDto(e);
        }
        return result;
    }
}
