package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("CloudAlert")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertDto extends EntityDto {
    @ApiModelProperty(
        value = "Optional description of the alert",
        position = 10)
    public String description;

    @ApiModelProperty(
        value = "State of the alert, only ACTIVE will be periodically checked",
        required = true,
        position = 11)
    public AlertState state;

    @ApiModelProperty(
        value = "Alert type specific configuration",
        required = true,
        position = 12)
    public AlertTypeDto type;

    @ApiModelProperty(
        value = "Notification channels that will receive events",
        required = true,
        position = 13)
    public List<AssociatedChannelDto> channels;

    @ApiModelProperty(
        value = "Notification channels properties",
        readOnly = true,
        position = 14)
    public List<NotificationChannelPropertiesWithRecipientsDto> channelProps;

    @ApiModelProperty(
        value = "Alert annotations",
        position = 15)
    public Map<String, String> annotations;

    @ApiModelProperty(
        value = "Alert window width in seconds. " +
                "Can't change in alert from template, instead use field in template",
        position = 16)
    public Long windowSecs;

    @ApiModelProperty(
        value = "Alert delay in seconds. " +
                "Can't change in alert from template, instead use field in template",
        position = 17)
    public Integer delaySecs;

    @ApiModelProperty(
            value = "Alert labels.",
            position = 18)
    public Map<String, String> labels;

    @ApiModelProperty(
            value = "Templates that explain alert evaluation status, and what todo when alert occurs. " +
                    "Variables available to use into template depends on alert type.",
            position = 19)
    public Map<String, String> serviceProviderAnnotations;

    @Nonnull
    public static AlertDto fromProto(@Nonnull TAlert alert) {
        return fromProto(alert, null);
    }

    @Nonnull
    public static AlertDto fromProto(@Nonnull TAlert alert, List<NotificationChannelPropertiesWithRecipientsDto> channelProps) {
        var ret = new AlertDto();

        EntityDto.fillFromProto(ret, alert);

        ret.description = alert.getDescription();
        ret.state = AlertState.fromProto(alert.getState());
        ret.type = AlertTypeDto.fromProto(alert);
        ret.channels = alert.getConfiguredNotificationChannelsMap().entrySet().stream()
            .map(e -> AssociatedChannelDto.of(e.getKey(), ChannelConfigDto.fromProto(e.getValue())))
            .collect(Collectors.toList());
        ret.annotations = alert.getAnnotationsMap();
        ret.serviceProviderAnnotations = alert.getServiceProviderAnnotationsMap();
        ret.windowSecs = alert.getPeriodMillis() / 1000L;
        ret.delaySecs = alert.getDelaySeconds();
        ret.labels = alert.getLabelsMap();

        ret.channelProps = channelProps;

        return ret;
    }

    public TAlert toProto(@Nonnull String projectId) {
        validate();

        var builder = TAlert.newBuilder()
            .setId(Nullables.orEmpty(id))
            .setProjectId(projectId)
            .setFolderId(folderId)
            .setVersion(Nullables.orZero(version))
            .setName(name)
            .setDescription(Nullables.orEmpty(description))
            .setState(Nullables.orDefault(state, AlertState.ACTIVE).toProto())
            .setPeriodMillis(Nullables.orZero(windowSecs) * 1000L)
            .setDelaySeconds(Nullables.orZero(delaySecs))
            .setCreatedBy(Nullables.orEmpty(createdBy))
            .setUpdatedBy(Nullables.orEmpty(updatedBy))
            .putAllAnnotations(Nullables.orEmpty(annotations))
            .putAllServiceProviderAnnotations(Nullables.orEmpty(serviceProviderAnnotations))
            .putAllConfiguredNotificationChannels(Nullables.orEmpty(channels).stream()
                .collect(Collectors.toMap(channel -> channel.id, channel -> channel.config.toProto())))
            .putAllLabels(Nullables.orEmpty(labels));

        if (createdAt != null) {
            builder.setCreatedAt(Instant.parse(createdAt).toEpochMilli());
        }
        if (updatedAt != null) {
            builder.setUpdatedAt(Instant.parse(updatedAt).toEpochMilli());
        }

        type.addToProto(builder, projectId, folderId);

        return builder.build();
    }

    private void validate() {
        Validators.checkPresent(name, "name");
        Validators.checkPresent(folderId, "folderId");

        Validators.checkPresent(type, "type");

        if (type.thresholdAlert != null) {
            Validators.checkPresent(windowSecs, "windowSecs");
            Validators.checkPositive(windowSecs, "windowSecs");
        }
        if (delaySecs != null) {
            Validators.checkNonNegative(delaySecs, "delaySecs");
        }
        Validators.checkNonEmpty(name, "name");
        Validators.checkNonEmpty(folderId, "folderId");
    }
}
