package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;

import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.expression.expr.func.util.SelFnAlertEvaluationHistory;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metrics.client.ReadResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("AlertEvaluationHistory")
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class AlertEvaluationHistoryDto {
    private static final int NO_DATA = TEvaluationStatus.ECode.NO_DATA_VALUE;
    private static final int OK = TEvaluationStatus.ECode.OK_VALUE;
    private static final int ERROR = TEvaluationStatus.ECode.ERROR_VALUE;
    private static final int ALARM = TEvaluationStatus.ECode.ALARM_VALUE;
    private static final int WARN = TEvaluationStatus.ECode.WARN_VALUE;

    @JsonProperty
    public long[] timestamps;
    @JsonProperty
    public long[] ok;
    @JsonProperty
    public long[] warn;
    @JsonProperty
    public long[] alarm;
    @JsonProperty
    public long[] noData;
    @JsonProperty
    public long[] error;
    @JsonProperty
    public long[] okMuted;
    @JsonProperty
    public long[] warnMuted;
    @JsonProperty
    public long[] alarmMuted;
    @JsonProperty
    public long[] noDataMuted;
    @JsonProperty
    public long[] errorMuted;

    public AlertEvaluationHistoryDto(int capacity) {
        this.timestamps = new long[capacity];
        this.ok = new long[capacity];
        this.warn = new long[capacity];
        this.alarm = new long[capacity];
        this.noData = new long[capacity];
        this.error = new long[capacity];
        this.okMuted = new long[capacity];
        this.warnMuted = new long[capacity];
        this.alarmMuted = new long[capacity];
        this.noDataMuted = new long[capacity];
        this.errorMuted = new long[capacity];
    }

    private void resize(int size) {
        this.timestamps = Arrays.copyOf(timestamps, size);
        this.ok = Arrays.copyOf(ok, size);
        this.warn = Arrays.copyOf(warn, size);
        this.alarm = Arrays.copyOf(alarm, size);
        this.noData = Arrays.copyOf(noData, size);
        this.error = Arrays.copyOf(error, size);
        this.okMuted = Arrays.copyOf(okMuted, size);
        this.warnMuted = Arrays.copyOf(warnMuted, size);
        this.alarmMuted = Arrays.copyOf(alarmMuted, size);
        this.noDataMuted = Arrays.copyOf(noDataMuted, size);
        this.errorMuted = Arrays.copyOf(errorMuted, size);
    }

    private void ensureCapacity(int size) {
        if (timestamps.length > size) {
            return;
        }

        int oldCapacity = timestamps.length;
        int newCapacity = oldCapacity + (oldCapacity >> 1);
        newCapacity = Math.max(newCapacity, size);
        resize(newCapacity);
    }

    public static AlertEvaluationHistoryDto create(ReadResponse readResponse, Interval interval, long gridMillis) {
        return create(readResponse, interval, gridMillis, false);
    }

    public static AlertEvaluationHistoryDto create(@Nonnull List<Metric<MetricKey>> metrics, Interval interval, long gridMillis) {
        var result = StatusMergingSupport.mergeFrom(metrics, false);
        int capacity = Math.toIntExact(interval.duration().toMillis() / gridMillis + 1);
        var dto = new AlertEvaluationHistoryDto(capacity);
        int index = 0;
        boolean skip = true;
        for (int i = 0; i < result.timestamps.length; i++) {
            if (skip) {
                skip = false;
                continue;
            } else {
                skip = true;
            }
            long tsMillis = result.timestamps[i];
            long prevTs = dto.timestamps[index];
            long truncated = InstantUtils.truncate(tsMillis, gridMillis);
            if (prevTs == 0) {
                dto.timestamps[index] = truncated;
            } else if (prevTs != truncated) {
                index++;
                dto.ensureCapacity(index + 1);
                dto.timestamps[index] = truncated;
            }
            dto.ok[index] += result.statuses.get("OK")[i];
            dto.warn[index] += result.statuses.get("WARN")[i];
            dto.alarm[index] += result.statuses.get("ALARM")[i];
            dto.noData[index] += result.statuses.get("NO_DATA")[i];
            dto.error[index] += result.statuses.get("ERROR")[i];
            dto.okMuted[index] += result.statuses.get("OK:MUTED")[i];
            dto.warnMuted[index] += result.statuses.get("WARN:MUTED")[i];
            dto.alarmMuted[index] += result.statuses.get("ALARM:MUTED")[i];
            dto.noDataMuted[index] += result.statuses.get("NO_DATA:MUTED")[i];
            dto.errorMuted[index] += result.statuses.get("ERROR:MUTED")[i];
        }
        dto.resize(index + 1);
        return dto;
    }

    public static AlertEvaluationHistoryDto create(ReadResponse readResponse, Interval interval, long gridMillis, boolean downsamplingOff) {
        if (!readResponse.isOk()) {
            return new AlertEvaluationHistoryDto(0);
        }

        var source = readResponse.getSource();
        var type = readResponse.getType();
        if (source.isEmpty()) {
            return new AlertEvaluationHistoryDto(0);
        }

        var it = MetricTypeTransfers.of(type, MetricType.IGAUGE, source.iterator());
        if (!it.hasColumn(StockpileColumn.TS) || !it.hasColumn(StockpileColumn.LONG_VALUE)) {
            throw new IllegalStateException("Not valid column set " + source.columnSet() + " to evaluation history");
        }

        int capacity = Math.toIntExact(interval.duration().toMillis() / gridMillis + 1);
        var dto = new AlertEvaluationHistoryDto(capacity);

        int index = 0;
        boolean skip = true;
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (it.next(point)) {
                if (!downsamplingOff && point.count == 0) {
                    continue;
                }
                if (downsamplingOff) {
                    if (skip) {
                        skip = false;
                        continue;
                    } else {
                        skip = true;
                    }
                }

                long prevTs = dto.timestamps[index];
                long truncated = InstantUtils.truncate(point.tsMillis, gridMillis);
                if (prevTs == 0) {
                    dto.timestamps[index] = truncated;
                } else if (prevTs != truncated) {
                    index++;
                    dto.ensureCapacity(index + 1);
                    dto.timestamps[index] = truncated;
                }

                int state = Math.toIntExact(point.longValue) % SelFnAlertEvaluationHistory.ALERT_STATUS_DIVISOR;
                switch (state) {
                    case OK -> {
                        dto.ok[index] += 1;
                        if (point.longValue > 100) {
                            dto.okMuted[index] += 1;
                        }
                    }
                    case ALARM -> {
                        dto.alarm[index] += 1;
                        if (point.longValue > 100) {
                            dto.alarmMuted[index] += 1;
                        }
                    }
                    case WARN -> {
                        dto.warn[index] += 1;
                        if (point.longValue > 100) {
                            dto.warnMuted[index] += 1;
                        }
                    }
                    case ERROR -> {
                        dto.error[index] += 1;
                        if (point.longValue > 100) {
                            dto.errorMuted[index] += 1;
                        }
                    }
                    case NO_DATA -> {
                        dto.noData[index] += 1;
                        if (point.longValue > 100) {
                            dto.noDataMuted[index] += 1;
                        }
                    }
                    default -> throw new IllegalStateException("Unexpected value: " + state);
                }
            }
        } finally {
            point.recycle();
        }

        dto.resize(index + 1);
        return dto;
    }
}
