package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.TDefaultChannelConfig;
import ru.yandex.solomon.alert.protobuf.TListAlert;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("AlertListItem")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertListItemDto {
    @ApiModelProperty(
        value = "Alert identifier",
        readOnly = true,
        dataType = "java.util.UUID",
        position = 0)
    public String id;

    @ApiModelProperty(
        value = "Human-readable name of the alert",
        readOnly = true,
        example = "My alert",
        position = 1)
    public String name;

    @ApiModelProperty(
        value = "State of the alert, only ACTIVE will be periodically checked",
        readOnly = true,
        position = 2)
    public AlertState state;

    @ApiModelProperty(
        value = "Type of the alert",
        readOnly = true,
        position = 3)
    public AlertType type;

    @ApiModelProperty(
        value = "Notification channels properties",
        readOnly = true,
        position = 4)
    public List<NotificationChannelPropertiesWithConfigDto> channelProps;

    @ApiModelProperty(
        value = "Current alert status",
        readOnly = true,
        position = 5)
    public EvaluationStatusDto.Code evaluationStatusCode;

    @ApiModelProperty(
        value = "Last status change timestamp",
        readOnly = true,
        dataType = "ISO-8601",
        position = 6)
    public String statusSince;

    public static AlertListItemDto fromProto(@Nonnull TListAlert listAlert, @Nonnull BiConsumer<String, NotificationChannelPropertiesDto> channelPropertiesResolver) {
        var dto = new AlertListItemDto();

        dto.id = listAlert.getId();
        dto.name = listAlert.getName();
        dto.state = AlertState.fromProto(listAlert.getAlertState());
        dto.type = AlertType.fromProto(listAlert.getAlertType());

        dto.channelProps = new ArrayList<>();
        for (int i = 0; i < listAlert.getNotificationChannelIdsCount(); i++) {
            NotificationChannelPropertiesWithConfigDto channelPropsDto = new NotificationChannelPropertiesWithConfigDto();
            channelPropertiesResolver.accept(listAlert.getNotificationChannelIds(i), channelPropsDto);
            if (listAlert.getConfiguredNotificationChannelsCount() > 0) {
                TNotificationChannelOptions config = listAlert.getConfiguredNotificationChannelsOrDefault(channelPropsDto.id,
                    TNotificationChannelOptions.newBuilder()
                        .setDefaultChannelConfig(TDefaultChannelConfig.getDefaultInstance())
                        .build());
                channelPropsDto.setConfig(ChannelConfigDto.fromProto(config));
            }
            dto.channelProps.add(channelPropsDto);
        }

        long statusSince = listAlert.getStatusSinceTimestamp();
        if (statusSince != 0L) {
            dto.evaluationStatusCode = EvaluationStatusDto.Code.fromProto(listAlert.getEvaluationStatusCode());
            dto.statusSince = Instant.ofEpochMilli(statusSince).toString();
        }

        return dto;
    }
}
