package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.core.label.LabelConf;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AllLabelValuesResponseDto {

    @JsonProperty("maxSensorsCount")
    private int maxMetricsCount;
    @JsonProperty("sensorsCountByCluster")
    private Map<String, Integer> metricsCountByCluster;
    @JsonProperty
    private List<LabelValuesDto> labels;

    @Nonnull
    public static AllLabelValuesResponseDto fromModel(
        @Nonnull LabelValuesResponse model,
        @Nonnull Selectors selectors,
        @Nonnull LabelValidationFilter validationFilter)
    {
        Set<String> exactKeys = selectors.stream()
            .filter(Selector::isExact)
            .map(Selector::getKey)
            .collect(Collectors.toSet());

        AllLabelValuesResponseDto dto = new AllLabelValuesResponseDto();
        LabelValuesStats labelValuesStats = model.getLabelValuesStats();

        dto.maxMetricsCount = model.getMetricsCountByDestination().isEmpty()
            ? 0
            : Collections.max(model.getMetricsCountByDestination().values());

        boolean hasLabelValidation = validationFilter != LabelValidationFilter.ALL;

        dto.labels = labelValuesStats.getStatsByLabelKey().entrySet()
            .stream()
            .filter(entry -> !LabelKeys.PROJECT.equals(entry.getKey()))
            .filter(entry -> hasLabelValidation || !entry.getValue().getValues().isEmpty())
            .filter(entry -> !exactKeys.contains(entry.getKey()))
            .sorted(Comparator.comparing(Map.Entry::getKey, LabelConf.getPartialSorterComparator()))
            .map(entry -> createLabelValues(labelValuesStats, entry))
            .collect(Collectors.toList());

        dto.metricsCountByCluster = model.getMetricsCountByDestination();
        return dto;
    }

    private static LabelValuesDto createLabelValues(
        @Nonnull LabelValuesStats labelValuesStats,
        @Nonnull Map.Entry<String, LabelStats> entry)
    {
        String name = entry.getKey();
        LabelStats labelStats = entry.getValue();
        boolean isAbsent = labelStats.getCount() < labelValuesStats.getMetricsCount();
        return LabelValuesDto.fromModel(name, labelStats, isAbsent);
    }
}
