package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.TChannelConfig;
import ru.yandex.solomon.alert.protobuf.TDefaultChannelConfig;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("ChannelConfig")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ChannelConfigDto {
    @ApiModelProperty(
        value = "Set of statuses to notify about",
        position = 0)
    public Set<EvaluationStatusDto.Code> notifyAboutStatuses;

    @ApiModelProperty(
        value = "Repeat notification delay in seconds. Zero means no repeating",
        position = 1)
    public Long repeatDelaySecs;

    @Nonnull
    public static ChannelConfigDto fromProto(@Nonnull TNotificationChannelOptions value) {
        switch (value.getOptionsCase()) {
            case DEFAULTCHANNELCONFIG:
                return new ChannelConfigDto();
            case CHANNELCONFIG:
                return fromProto(value.getChannelConfig());
            default:
                throw new UnsupportedOperationException("Unsupported channel config type: " + value.getOptionsCase());
        }
    }

    @Nonnull
    private static ChannelConfigDto fromProto(@Nonnull TChannelConfig channelConfig) {
        var ret = new ChannelConfigDto();
        ret.notifyAboutStatuses = channelConfig.getNotifyAboutStatusesList().stream()
            .map(EvaluationStatusDto.Code::fromProto)
            .collect(Collectors.toSet());
        ret.repeatDelaySecs = channelConfig.getRepeatNotifyDelayMillis() / 1000L;
        return ret;
    }

    public TNotificationChannelOptions toProto() {
        if (notifyAboutStatuses == null) {
            return TNotificationChannelOptions.newBuilder()
                .setDefaultChannelConfig(TDefaultChannelConfig.getDefaultInstance())
                .build();
        }
        return TNotificationChannelOptions.newBuilder()
            .setChannelConfig(TChannelConfig.newBuilder()
                .addAllNotifyAboutStatuses(notifyAboutStatuses.stream()
                    .map(EvaluationStatusDto.Code::toProto)
                    .collect(Collectors.toList()))
                .setRepeatNotifyDelayMillis(Nullables.orZero(repeatDelaySecs) * 1000L)
            )
            .build();
    }
}
