package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.time.Instant;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("CloudNotificationChannel")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class NotificationChannelDto extends EntityDto {
    @ApiModelProperty(
        value = "Channel transport type",
        required = true,
        position = 10)
    public TransportType type;

    // The following fields follow type.name().toLowerCase() naming convention

    @ApiModelProperty(
        value = "Email-specific transport settings",
        position = 11)
    public EmailChannelSettingsDto email;

    @ApiModelProperty(
        value = "Sms-specific transport settings",
        position = 12)
    public SmsChannelSettingsDto sms;

    @ApiModelProperty(
            value = "Push-specific transport settings",
            position = 13)
    public PushChannelSettingsDto push;

    @ApiModelProperty(
            value = "Default channel for project",
            position = 14)
    public boolean isDefault;

    @ApiModelProperty(
            value = "Channel is default for alert severity.",
            position = 14)
    public List<Severity> defaultForAlertSeverity;

    public static NotificationChannelDto fromProto(@Nonnull TNotification notification) {
        var dto = new NotificationChannelDto();

        EntityDto.fillFromProto(dto, notification);

        dto.isDefault = notification.getDefaultForProject();
        dto.type = TransportType.fromProto(notification.getTypeCase());
        dto.defaultForAlertSeverity = notification.getDefaultForAlertSeverityList();
        switch (notification.getTypeCase()) {
            case CLOUDSMS -> dto.sms = SmsChannelSettingsDto.fromProto(notification.getCloudSms());
            case CLOUDEMAIL -> dto.email = EmailChannelSettingsDto.fromProto(notification.getCloudEmail());
            case CLOUDPUSH -> dto.push = PushChannelSettingsDto.fromProto(notification.getCloudPush());
        }

        return dto;
    }

    public TNotification toProto(@Nonnull String projectId) {
        validate();

        var builder = TNotification.newBuilder()
            .setId(Nullables.orEmpty(id))
            .setProjectId(projectId)
            .setFolderId(folderId)
            .setVersion(Nullables.orZero(version))
            .setName(name)
            .setCreatedBy(Nullables.orEmpty(createdBy))
            .setUpdatedBy(Nullables.orEmpty(updatedBy))
            .setDefaultForProject(!Nullables.orEmpty(defaultForAlertSeverity).isEmpty() || isDefault)
            .addAllDefaultForAlertSeverity(Nullables.orEmpty(defaultForAlertSeverity))
            ;

        if (createdAt != null) {
            builder.setCreatedAt(Instant.parse(createdAt).toEpochMilli());
        }
        if (updatedAt != null) {
            builder.setUpdatedAt(Instant.parse(updatedAt).toEpochMilli());
        }

        switch (type) {
            case EMAIL -> builder.setCloudEmail(email.toProto());
            case SMS -> builder.setCloudSms(sms.toProto());
            case PUSH -> builder.setCloudPush(push.toProto());
        }

        return builder.build();
    }

    private void validate() {
        Validators.checkPresent(name, "name");
        Validators.checkPresent(folderId, "folderId");

        Validators.checkPresent(type, "type");

        switch (type) {
            case EMAIL -> Validators.checkPresent(email, "email");
            case SMS -> Validators.checkPresent(sms, "sms");
            case PUSH -> Validators.checkPresent(push, "push");
            default -> throw new BadRequestException("Invalid type parameter: " + type);
        }

        Validators.checkNonEmpty(name, "name");
        Validators.checkNonEmpty(folderId, "folderId");
    }
}
