package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.solomon.model.timeseries.iterator.AggrPointCursor;
import ru.yandex.solomon.model.timeseries.iterator.GenericCombineIterator;
import ru.yandex.solomon.model.timeseries.iterator.GenericPointCollector;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList.empty;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class StatusMergingSupport {

    private static class GrowingLongArray {
        private long[] data;
        private int size;

        public GrowingLongArray(int capacity) {
            data = new long[capacity];
            size = 0;
        }

        private void grow() {
            int capacity = data.length;
            int newCapacity = capacity + (capacity >> 1) + 1;
            data = Arrays.copyOf(data, newCapacity);
        }

        public void add(long value) {
            if (size == data.length) {
                grow();
            }
            data[size++] = value;
        }

        public long[] dropFirst() {
            if (size < 2) {
                return new long[0];
            }
            return Arrays.copyOfRange(data, 1, size);
        }

        public long[] inplaceDelta() {
            if (size < 2) {
                return new long[0];
            }
            for (int i = 0; i < size - 1; i++) {
                long diff = data[i + 1] - data[i];
                // FIXME: if diff < 0 then COUNTER was reset between ts[i] and ts[i+1]
                // Proper delta is not data[i+1] but data[i+1] + max(data) on [ts[i], ts[i+1]] - data[i]
                data[i] = diff >= 0 ? diff : data[i+1];
            }
            size--;
            return Arrays.copyOf(data, size);
        }
    }

    public static class MergeResult {
        public final long[] timestamps;
        public final Map<String, long[]> statuses;

        private MergeResult(GrowingLongArray timestamps, List<GrowingLongArray> statuses, List<String> statusKeys) {
            this(timestamps, statuses, statusKeys, true);
        }

        private MergeResult(GrowingLongArray timestamps, List<GrowingLongArray> statuses, List<String> statusKeys, boolean deltas) {
            this.timestamps = timestamps.dropFirst();

            this.statuses = IntStream.range(0, statusKeys.size())
                .filter(i -> !statusKeys.get(i).isEmpty())
                .mapToObj(i -> Map.entry(statusKeys.get(i), deltas ? statuses.get(i).inplaceDelta() : statuses.get(i).dropFirst()))
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
        }

        public int size() {
            return timestamps.length;
        }
    }

    public static MergeResult mergeFrom(List<Metric<MetricKey>> metrics) {
        return mergeFrom(metrics, true);
    }

    public static MergeResult mergeFrom(List<Metric<MetricKey>> metrics, boolean deltas) {
        List<String> statusKeys = metrics.stream()
            .map(metric -> Optional.ofNullable(metric.getKey())
                .map(key -> key.getLabels().findByKey("status"))
                .map(Label::getValue)
                .orElse(""))
            .collect(toList());

        int pointsEstimate = metrics.stream()
            .map(Metric::getTimeseries)
            .filter(Objects::nonNull)
            .map(AggrGraphDataIterable::getRecordCount)
            .reduce(Integer::max)
            .orElse(0);

        List<AggrGraphDataListIterator> iterators = metrics.stream()
            .map(metric -> {
                var iterable = metric.getTimeseries();
                var type = metric.getType();
                if (iterable == null || iterable.isEmpty()) {
                    return empty().iterator();
                }
                return MetricTypeTransfers.of(type, MetricType.IGAUGE, iterable.iterator());
            })
            .collect(Collectors.toList());

        return mergeFrom(statusKeys, iterators, pointsEstimate, deltas);
    }

    private static class StatusAppender implements GenericPointCollector<AggrPointCursor, Object> {
        private final GrowingLongArray timestamps;
        private final List<GrowingLongArray> statuses;
        private final long[] values;

        StatusAppender(GrowingLongArray timestamps, List<GrowingLongArray> statuses) {
            this.timestamps = timestamps;
            this.statuses = statuses;
            this.values = new long[statuses.size()];
        }

        @Override
        public void reset() {
            // no op
        }

        @Override
        public void append(int cursorIndex, AggrPointCursor cursor) {
            values[cursorIndex] = cursor.getPoint().longValue;
        }

        @Override
        public void compute(long nextTsMillis, Object ignore) {
            timestamps.add(nextTsMillis);
            for (int i = 0; i < statuses.size(); i++) {
                statuses.get(i).add(values[i]);
            }
        }
    }

    private static MergeResult mergeFrom(List<String> statusKeys, List<AggrGraphDataListIterator> iterators, int pointsEstimate, boolean deltas) {
        GrowingLongArray timestamps = new GrowingLongArray(pointsEstimate);
        List<AggrPointCursor> cursors = iterators.stream().map(AggrPointCursor::new).collect(toList());
        List<GrowingLongArray> statuses = cursors.stream()
                .map(ignore -> new GrowingLongArray(pointsEstimate))
                .collect(toList());
        var combineIterator = new GenericCombineIterator<>(
                cursors,
                new StatusAppender(timestamps, statuses)
        );

        while (combineIterator.next(null)) {
            // just move the iterator
        }

        return new MergeResult(timestamps, statuses, statusKeys, deltas);
    }
}
