package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Preconditions;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ApiModel("ThresholdAlertSettings")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ThresholdAlertDto {
    @ApiModelProperty(
        value = "Selectors that query data to be checked",
        required = true,
        position = 0)
    public String selectors;

    @ApiModelProperty(
        value = "Transformations to be applied to the data before checking",
        required = true,
        position = 1)
    public String transformations;

    @ApiModelProperty(
        value = "Function used to aggregate checks in the alert window by time",
        required = true,
        position = 2)
    public TimeAggregation timeAggregation;

    @ApiModelProperty(
        value = "Operation used to compare with the threshold value",
        required = true,
        position = 3)
    public Comparison comparison;

    @ApiModelProperty(
        value = "Alarm threshold value",
        required = true,
        position = 4)
    public Double alarmThreshold;

    @ApiModelProperty(
        value = "Warning threshold value",
        position = 5)
    public Double warningThreshold;

    @Nonnull
    public static ThresholdAlertDto fromProto(@Nonnull TThreshold threshold) {
        var ret = new ThresholdAlertDto();
        ret.selectors = Selectors.format(LabelSelectorConverter.protoToSelectors(threshold.getNewSelectors()));
        ret.transformations = threshold.getTransformations();
        var predicateRules = threshold.getPredicateRulesList();

        if (predicateRules.size() == 1) {
            // Only alarm threshold
            var rule = predicateRules.get(0);
            Preconditions.checkArgument(rule.getTargetStatus() == TPredicateRule.ETargetStatus.ALARM,
                "Single predicate rule must have ALARM target status");
            ret.timeAggregation = TimeAggregation.fromProto(rule.getThresholdType());
            ret.comparison = Comparison.fromProto(rule.getComparison());
            ret.alarmThreshold = rule.getThreshold();
            ret.warningThreshold = null;
            return ret;
        }
        if (predicateRules.size() == 2) {
            // Alarm + Warning threshold
            var alarmRule = predicateRules.get(0);
            var warnRule = predicateRules.get(1);
            Preconditions.checkArgument(alarmRule.getTargetStatus() == TPredicateRule.ETargetStatus.ALARM,
                "First predicate rule must have ALARM target status");
            Preconditions.checkArgument(warnRule.getTargetStatus() == TPredicateRule.ETargetStatus.WARN,
                "Second predicate rule must have WARN target status");
            Preconditions.checkArgument(warnRule.getComparison() == alarmRule.getComparison(),
                "Rules differ in comparison functions");
            Preconditions.checkArgument(warnRule.getThresholdType() == alarmRule.getThresholdType(),
                "Rules differ in aggregation function");
            ret.timeAggregation = TimeAggregation.fromProto(warnRule.getThresholdType());
            ret.comparison = Comparison.fromProto(warnRule.getComparison());
            ret.alarmThreshold = alarmRule.getThreshold();
            ret.warningThreshold = warnRule.getThreshold();
            return ret;
        }

        throw new IllegalArgumentException("Unsupported alert with more than two rules");
    }

    public void addToProto(@Nonnull TAlert.Builder builder, String cloudId, String folderId) {
        validate();

        Selectors parsedSelectors;
        try {
            parsedSelectors = Selectors.parse(selectors).toBuilder()
                    .addOverride(LabelKeys.PROJECT, cloudId)
                    .addOverride(LabelKeys.CLUSTER, folderId)
                    .build();
        } catch (RuntimeException e) {
            throw new BadRequestException("Bad \"selectors\" parameter: " + e.getMessage());
        }

        TThreshold.Builder threshold = TThreshold.newBuilder()
            .setNewSelectors(LabelSelectorConverter.selectorsToNewProto(parsedSelectors))
            .setTransformations(Nullables.orEmpty(transformations));

        if (warningThreshold == null) {
            threshold.addPredicateRules(TPredicateRule.newBuilder()
                .setComparison(comparison.toProto())
                .setThresholdType(timeAggregation.toProto())
                .setThreshold(alarmThreshold)
                .setTargetStatus(TPredicateRule.ETargetStatus.ALARM)
                .build());
        } else {
            validateThresholds(warningThreshold, alarmThreshold, comparison);

            threshold.addPredicateRules(TPredicateRule.newBuilder()
                .setComparison(comparison.toProto())
                .setThresholdType(timeAggregation.toProto())
                .setThreshold(alarmThreshold)
                .setTargetStatus(TPredicateRule.ETargetStatus.ALARM)
                .build());
            threshold.addPredicateRules(TPredicateRule.newBuilder()
                .setComparison(comparison.toProto())
                .setThresholdType(timeAggregation.toProto())
                .setThreshold(warningThreshold)
                .setTargetStatus(TPredicateRule.ETargetStatus.WARN)
                .build());
        }

        builder.setThreshold(threshold.build());
    }

    private void validate() {
        Validators.checkPresent(selectors, "selectors");
        Validators.checkPresent(comparison, "comparison");
        Validators.checkPresent(timeAggregation, "timeAggregation");
        Validators.checkPresent(alarmThreshold, "alarmThreshold");

        Validators.checkKnown(comparison, "comparison");
        Validators.checkKnown(timeAggregation, "timeAggregation");
    }

    private void validateThresholds(double warningThreshold, double alarmThreshold, Comparison comparison) {
        if (comparison == Comparison.GT || comparison == Comparison.GTE) {
            if (warningThreshold >= alarmThreshold) {
                throw new BadRequestException("the alarm threshold must be greater than the warning threshold");
            }
            return;
        }
        if (comparison == Comparison.LT || comparison == Comparison.LTE) {
            if (warningThreshold <= alarmThreshold) {
                throw new BadRequestException("the alarm threshold must be lesser than the warning threshold");
            }
            return;
        }
        if (warningThreshold == alarmThreshold) {
            throw new BadRequestException("the alarm threshold and the warning threshold must differ");
        }
    }
}
