package ru.yandex.solomon.gateway.api.cloud.v2;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
class FolderServiceMetrics {
    final String cloudId;
    final String folderId;
    final String service;
    private final MetricRegistry registry;
    private final AsyncMetrics asyncMetrics;
    private final Histogram responseSizeBytes;
    private final Rate responseSizeBytesFlow;
    private final Rate gauges;
    private final Rate counters;
    private final Rate histograms;

    public FolderServiceMetrics(String cloudId, String folderId, String serviceId) {
        this.cloudId = cloudId;
        this.folderId = folderId;
        this.service = serviceId;
        this.registry = new MetricRegistry(Labels.of("cloudId", cloudId, "folderId", folderId, "serviceId", serviceId));
        this.asyncMetrics = new AsyncMetrics(registry, "prometheusClient");
        this.responseSizeBytes = registry.histogramRate("prometheusClient.responseSizeBytes", Histograms.exponential(15, 2, 65536));
        this.responseSizeBytesFlow = registry.rate("prometheusClient.responseSizeBytesFlow");
        this.gauges = metricTypeMetric("GAUGE");
        this.counters = metricTypeMetric("COUNTER");
        this.histograms = metricTypeMetric("HISTOGRAM");
    }

    private Rate metricTypeMetric(String gauge) {
        return registry.rate("prometheusClient.metricTypes", Labels.of("type", gauge));
    }

    void callStarted() {
        asyncMetrics.callStarted();
    }

    void callCompleted(long durationNanos) {
        asyncMetrics.callCompletedOk(TimeUnit.NANOSECONDS.toMillis(durationNanos));
    }

    void callFailed(long durationNanos) {
        asyncMetrics.callCompletedError(TimeUnit.NANOSECONDS.toMillis(durationNanos));
    }

    void registerResponseSize(int size) {
        responseSizeBytes.record(size);
        responseSizeBytesFlow.add(size);
    }

    void combine(FolderServiceMetrics other) {
        this.asyncMetrics.combine(other.asyncMetrics);
        this.responseSizeBytes.combine(other.responseSizeBytes);
        this.responseSizeBytesFlow.combine(other.responseSizeBytesFlow);
        this.gauges.combine(other.gauges);
        this.counters.combine(other.counters);
        this.histograms.combine(other.histograms);
    }

    void registerGauge() {
        this.gauges.inc();
    }

    void registerCounter() {
        this.counters.inc();
    }

    void registerHistogram() {
        this.histograms.inc();
    }

    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }
}
