package ru.yandex.solomon.gateway.api.cloud.v2;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.google.common.base.Strings;
import io.netty.buffer.Unpooled;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.encode.spack.format.SpackHeader;
import ru.yandex.monlib.metrics.encode.spack.format.SpackVersion;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.conf.ServicesManager;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.ext.ExternalMonitoringMetrics;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudAuthorizer;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ReadRequestDto;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ReadResultDto;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.WriteResultDto;
import ru.yandex.solomon.gateway.cloud.api.RequestProducerResolver;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.push.PushFormats;
import ru.yandex.solomon.gateway.push.PushMetricProcessor;
import ru.yandex.solomon.gateway.push.PushRequest;
import ru.yandex.solomon.gateway.push.PushStatusToHttpStatus;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.util.time.Deadline;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;

/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
@RestController
@RequestMapping(path = "/monitoring/v2/data", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ ServicesManager.class })
public class MonitoringDataController {

    private static final Logger logger = LoggerFactory.getLogger(MonitoringDataController.class);

    /**
     * Service names for user specific metrics.
     */
    private static final String CUSTOM_METRICS_SERVICE = "custom";
    private static final String DATAPROC_METRICS_SERVICE = "data-proc";
    public static final Set<String> PUBLIC_SERVICES = Set.of(CUSTOM_METRICS_SERVICE, DATAPROC_METRICS_SERVICE);

    private final CloudAuthorizer authorizer;
    private final PushMetricProcessor pushMetricProcessor;
    private final DataClient dataClient;
    private final RequestProducerResolver producerResolver;

    @Autowired
    public MonitoringDataController(
            CloudAuthorizer authorizer,
            PushMetricProcessor pushMetricProcessor,
            DataClient dataClient,
            RequestProducerResolver producerResolver)
    {
        this.authorizer = authorizer;
        this.pushMetricProcessor = pushMetricProcessor;
        this.dataClient = dataClient;
        this.producerResolver = producerResolver;
    }

    private static Permission permissionForService(String service) {
        if (DATAPROC_METRICS_SERVICE.equals(service)) {
            return Permission.DATA_WRITE_DATAPROC;
        }
        return Permission.DATA_WRITE;
    }

    @RequestMapping(path = "/write", method = RequestMethod.POST)
    public CompletableFuture<ResponseEntity<WriteResultDto>> write(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam("service") String service,
        @RequestBody byte[] content,
        ServerHttpRequest httpRequest)
    {
        Permission permission = permissionForService(service);

        return authorizer.authorizeAndResolveCloudId(subject, folderId, permission, (account, cloudId) -> {
            var future = writeImpl(cloudId, folderId, service, content, httpRequest, account);
            ExternalMonitoringMetrics.forFuture("/v2/data/write", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ResponseEntity<WriteResultDto>> writeImpl(
            String cloudId,
            String folderId,
            String service,
            byte[] content,
            ServerHttpRequest httpRequest,
            Account account)
    {
        if (Strings.isNullOrEmpty(service)) {
            throw new BadRequestException("service parameter cannot be empty");
        }

        if (!authorizer.isGodUser(account.getId()) && !PUBLIC_SERVICES.contains(service)) {
            throw new BadRequestException("invalid service name (" + service + ") for custom metrics");
        }

        final String contentType = httpRequest.getHeaders().getFirst(CONTENT_TYPE.toString());
        if (StringUtils.isEmpty(contentType)) {
            throw new BadRequestException(CONTENT_TYPE.toString() + " header is missing");
        }

        final MetricFormat format = PushFormats.byMonitoringContentType(contentType);
        if (format == MetricFormat.METRIC_FORMAT_UNSPECIFIED) {
            String msg = "cannot determine metrics format by given content type: '" + contentType + '\'';
            throw new BadRequestException(msg);
        }

        if (format == MetricFormat.SPACK) {
            var header = SpackHeader.readFrom(ByteBuffer.wrap(content));
            if (header.getVersion().lt(SpackVersion.v1_2)) {
                throw new BadRequestException(
                        "unsupported spack version " + header.getVersion() +
                        ", expected at least " + SpackVersion.v1_2);
            }
        }

        final var shardKey = new ShardKey(cloudId, folderId, service);
        final var request = new PushRequest(shardKey, format, Unpooled.wrappedBuffer(content), content.length, account);

        return pushMetricProcessor.processPush(request, true)
            .whenComplete((r, t) -> {
                if (t != null) {
                    Throwable cause = CompletableFutures.unwrapCompletionException(t);
                    if (cause instanceof UnknownShardException) {
                        logger.warn("write into unknown shard {}", shardKey);
                    } else if (cause instanceof CompletionException) {
                        logger.warn("write into {} failed: {}", shardKey, t.getMessage());
                    } else {
                        logger.warn("write into {} failed", shardKey, t);
                    }
                }
            })
            .thenApply(r -> ResponseEntity.status(PushStatusToHttpStatus.convert(r.getStatus()))
                .body(new WriteResultDto(r.getSuccessMetricCount(), r.getErrorMessage())));
    }

    @RequestMapping(path = "/read", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    CompletableFuture<ReadResultDto> read(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestBody ReadRequestDto requestDto,
        @RequestHeader(name = "x-ui-request", required = false, defaultValue = "") String uiHeader)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.DATA_READ, cloudId -> {
            var future = readImpl(cloudId, folderId, requestDto, uiHeader, subject);
            ExternalMonitoringMetrics.forFuture("/v2/data/write", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ReadResultDto> readImpl(
            String cloudId,
            String folderId,
            ReadRequestDto requestDto,
            String uiHeader,
            AuthSubject subject)
    {
        Instant deadline = Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
        requestDto.validate();
        RequestProducer producer = producerResolver.resolve(uiHeader);
        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());
        DataRequest request = requestDto.toDataRequest(cloudId, folderId, producer, deadline, subjectId);
        return dataClient.readData(request)
            .thenApply(ReadResultDto::fromModel);
    }
}
