package ru.yandex.solomon.gateway.api.cloud.v2;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.cloud.ext.ExternalMonitoringMetrics;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudAuthorizer;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ListLabelKeysResultDto;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ListLabelValuesResultDto;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ListMetricNamesResultDto;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.ListMetricsResultDto;
import ru.yandex.solomon.gateway.api.v2.managers.MetricsManager;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
@RestController
@RequestMapping(path = "/monitoring/v2/metrics", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Import({ MetricsManager.class})
public class MonitoringMetaController {
    private static final LabelValidationFilter validationFilter = LabelValidationFilter.ALL;

    @Autowired
    private CloudAuthorizer authorizer;
    @Autowired
    private MetricsManager metricsManager;


    @RequestMapping(method = RequestMethod.GET)
    CompletableFuture<ListMetricsResultDto> listMetrics(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "pageSize", defaultValue = "30", required = false) int pageSize,
        @RequestParam(value = "pageToken", defaultValue = "", required = false) String pageToken)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRICS_GET, cloudId -> {
            var future = listMetricsImpl(cloudId, folderId, selectorsStr, pageSize);
            ExternalMonitoringMetrics.forFuture("/v2/metrics", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ListMetricsResultDto> listMetricsImpl(
            String cloudId,
            String folderId,
            String selectorsStr,
            int pageSize)
    {
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager.findMetrics(selectors, "", new PageOptions(pageSize, 0), true)
            .thenApply(metrics -> ListMetricsResultDto.fromModel(metrics.getResult()));
    }

    @RequestMapping(path = "/names", method = RequestMethod.GET)
    CompletableFuture<ListMetricNamesResultDto> listMetricNames(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "limit", defaultValue = "0") int limit)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_LABELS_GET, cloudId -> {
            var future = listMetricNamesImpl(cloudId, folderId, selectorsStr, text, limit);
            ExternalMonitoringMetrics.forFuture("/v2/metrics/names", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ListMetricNamesResultDto> listMetricNamesImpl(
            String cloudId,
            String folderId,
            String selectorsStr,
            String text,
            int limit) {
        Instant deadline = deadline();
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager.findMetricNames(selectors, text, LabelValidationFilter.ALL, limit, "", deadline)
                .thenApply(ListMetricNamesResultDto::fromModel);
    }

    @RequestMapping(path = "/labels", method = RequestMethod.GET)
    CompletableFuture<ListLabelKeysResultDto> listLabelKeys(
        @RequireAuth AuthSubject subject,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_NAMES_GET, cloudId -> {
            var future = listLabelKeysImpl(cloudId, folderId, selectorsStr);
            ExternalMonitoringMetrics.forFuture("/v2/labels", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ListLabelKeysResultDto> listLabelKeysImpl(
            String cloudId,
            String folderId,
            String selectorsStr) {
        Instant deadline = deadline();
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager.findLabelNames(selectors, true, "", deadline)
                .thenApply(ListLabelKeysResultDto::fromModel);
    }

    @RequestMapping(path = "/labels/{key}/values", method = RequestMethod.GET)
    CompletableFuture<ListLabelValuesResultDto> listLabelValues(
        @RequireAuth AuthSubject subject,
        @PathVariable("key") String labelKey,
        @RequestParam("folderId") String folderId,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr,
        @RequestParam(value = "text", defaultValue = "") String text,
        @RequestParam(value = "limit", defaultValue = "1000") int limit)
    {
        return authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.METRIC_LABELS_GET, cloudId -> {
            var future = listLabelValuesImpl(cloudId, folderId, labelKey, selectorsStr, text, limit);
            ExternalMonitoringMetrics.forFuture("/v2/labels/:key/values", cloudId, folderId, future);
            return future;
        });
    }

    private CompletableFuture<ListLabelValuesResultDto> listLabelValuesImpl(
            String cloudId,
            String folderId,
            String labelKey,
            String selectorsStr,
            String text,
            int limit) {
        Instant deadline = deadline();
        Selectors selectors = parseSelectors(cloudId, folderId, selectorsStr);
        return metricsManager
                .findLabelValues(List.of(labelKey), selectors, text, validationFilter, limit, true, "", deadline)
                .thenApply(model -> ListLabelValuesResultDto.fromModel(model, validationFilter));
    }

    private static Selectors parseSelectors(String cloudId, String folderId, String selectors) {
        try {
            return Selectors.parse(selectors)
                .toBuilder()
                .addOverride(Selector.exact(LabelKeys.PROJECT, cloudId))
                .addOverride(Selector.exact(LabelKeys.CLUSTER, folderId))
                .build();
        } catch (Throwable e) {
            throw new BadRequestException(String.format("failed to parse %s", selectors));
        }
    }

    private static Instant deadline() {
        return Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
    }
}
