package ru.yandex.solomon.gateway.api.cloud.v2;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.model.MetricKey;

/**
 * @author Vladimir Gordiychuk
 */
public class PagedDataLoader {
    private static final int LOOKBACK_DELTA_MINUTES = 5;
    private static final int METRICS_PAGE_LIMIT = 10000;

    private final MetricsClient client;
    private final List<MetricKey> metrics;
    private final Instant now;
    private final Instant deadline;
    private final RequestProducer producer;
    private final String subjectId;
    private final CompletableFuture<List<Metric<MetricKey>>> doneFuture = new CompletableFuture<>();
    private final List<Metric<MetricKey>> result;
    private int offset;

    public PagedDataLoader(MetricsClient client, List<MetricKey> metrics, Instant now, Instant deadline, RequestProducer producer, String subjectId) {
        this.client = client;
        this.metrics = metrics;
        this.producer = producer;
        this.result = new ArrayList<>(metrics.size());
        this.now = now;
        this.deadline = deadline;
        this.subjectId = subjectId;
    }

    public CompletableFuture<List<Metric<MetricKey>>> read() {
        if (metrics.isEmpty()) {
            return CompletableFuture.completedFuture(List.of());
        }

        loadNextPage();
        return doneFuture;
    }

    private void loadNextPage() {
        var req = ReadManyRequest.newBuilder()
                .addKeys(metrics.subList(offset, Math.min(offset + METRICS_PAGE_LIMIT, metrics.size())))
                .setFrom(now.minus(LOOKBACK_DELTA_MINUTES, ChronoUnit.MINUTES))
                .setTo(now)
                .setDeadline(deadline)
                .setProducer(producer)
                .setSubjectId(subjectId)
                .addOperation(Operation.newBuilder()
                        .setSummary(OperationAggregationSummary.newBuilder()
                                .addAggregations(Aggregation.LAST)
                                .build())
                        .build())
                .addOperation(Operation.newBuilder()
                        .setDropTimeseries(OperationDropTimeSeries.newBuilder().build())
                        .build())
                .build();

        client.readMany(req).whenComplete(this::onPageLoad);
    }

    private void onPageLoad(ReadManyResponse resp, @Nullable Throwable e) {
        if (e != null) {
            doneFuture.completeExceptionally(e);
            return;
        }

        if (!resp.isOk()) {
            doneFuture.completeExceptionally(new StockpileClientException(resp.getStatus()));
            return;
        }

        result.addAll(resp.getMetrics());
        offset += METRICS_PAGE_LIMIT;
        if (offset < metrics.size()) {
            loadNextPage();
        } else {
            doneFuture.complete(result);
        }
    }
}
