package ru.yandex.solomon.gateway.api.cloud.v2;

import java.io.ByteArrayOutputStream;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.google.common.primitives.Doubles;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.prometheus.ExtendedPrometheusWriter;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.cloud.billing.BillingLog;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.Int64Summary;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.collection.enums.EnumMapToInt;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ConditionalOnBean(TGatewayCloudConfig.class)
@ParametersAreNonnullByDefault
public class PrometheusMetricsClient {
    private static final Logger logger = LoggerFactory.getLogger(PrometheusMetricsClient.class);
    private static final RequestProducer PRODUCER = RequestProducer.PROMETHEUS_EXPORTER;

    private final MetricsClient metricsClient;
    private final SolomonConfHolder confHolder;
    private final PrometheusMetricsClientMetrics metrics;
    private final BillingLog billingLog;
    private final PagedMetaLoader metaLoader;

    @Autowired
    public PrometheusMetricsClient(MetricsClient metricsClient, SolomonConfHolder confHolder, BillingLog billingLog, PrometheusMetricsClientMetrics metrics) {
        this.metricsClient = metricsClient;
        this.confHolder = confHolder;
        this.metrics = metrics;
        this.billingLog = billingLog;
        this.metaLoader = new PagedMetaLoader(metricsClient);
    }

    public CompletableFuture<byte[]> loadPrometheusMetrics(
        String cloudId,
        String folderId,
        String service,
        String selectorsStr,
        Instant now,
        Instant deadline,
        String subjectId)
    {
        String validation = validate(cloudId, folderId, service);
        if (validation != null) {
            return CompletableFuture.failedFuture(new BadRequestException(validation));
        }

        try {
            var shard = confHolder.getConfOrThrow().findShardOrNull(new ShardKey(cloudId, folderId, service));
            if (shard == null) {
                return CompletableFuture.completedFuture(new byte[0]);
            }
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }

        FolderServiceMetrics folderServiceMetrics = metrics.getFolderServiceMetrics(cloudId, folderId, service);
        folderServiceMetrics.callStarted();
        long startTimeNanos = System.nanoTime();

        Selectors selectors = parseSelectors(cloudId, folderId, service, selectorsStr);

        return metaLoader.find(selectors, deadline, PRODUCER)
                .thenCompose(keys -> new PagedDataLoader(metricsClient, keys, now, deadline, PRODUCER, subjectId).read())
                .thenApply(metrics -> encode(metrics, folderServiceMetrics, now))
                .whenComplete((response, throwable) -> {
                    long endTimeNanos = System.nanoTime();
                    long durationNanos = endTimeNanos - startTimeNanos;

                    if (throwable != null) {
                        folderServiceMetrics.callFailed(durationNanos);
                        logger.warn("failed to implement prometheus metrics request", throwable);
                    } else {
                        folderServiceMetrics.callCompleted(durationNanos);
                        folderServiceMetrics.registerResponseSize(response.length);
                    }
                });
    }

    @Nullable
    private String validate(String cloudId, String folderId, String service) {
        if (!LabelsValidator.isValueValid(cloudId)) {
            return "invalid cloudId";
        }
        if (!LabelsValidator.isValueValid(folderId)) {
            return "invalid folderId";
        }
        if (!LabelsValidator.isValueValid(service)) {
            return "invalid service";
        }
        return null;
    }

    private byte[] encode(List<Metric<MetricKey>> metrics, FolderServiceMetrics serviceMetrics, Instant now) {
        ByteArrayOutputStream out = new ByteArrayOutputStream(24 << 10); // 24 KiB
        Table<MetricId, Double, Double> histogramMetrics = HashBasedTable.create(100, 20);
        EnumMapToInt<MetricType> stats = new EnumMapToInt<>(MetricType.class);
        try (ExtendedPrometheusWriter writer = new ExtendedPrometheusWriter(out)) {
            var labelsBuilder = Labels.builder();

            for (var metric : metrics) {
                var metricKey = metric.getKey();
                if (metricKey == null) {
                    continue;
                }

                TimeseriesSummary summary = metric.getSummary();
                if (summary == null || summary.getCount() == 0) {
                    continue;
                }

                final double value;
                if (summary instanceof DoubleSummary) {
                    value = ((DoubleSummary) summary).getLast();
                } else if (summary instanceof Int64Summary) {
                    value = ((Int64Summary) summary).getLast();
                } else {
                    continue;
                }

                stats.incrementAndGet(metricKey.getType());
                var name = metricKey.getName();
                var labels = metricKey.getLabels();

                labelsBuilder.clear();
                labelsBuilder
                        .addAll(labels)
                        .remove(LabelKeys.PROJECT)
                        .remove(LabelKeys.CLUSTER)
                        .remove(LabelKeys.SERVICE);

                var binLabel = labels.findByKey(LabelKeys.BIN);

                if (binLabel != null) {
                    String binValue = binLabel.getValue();
                    Double bound = parseBinValue(binValue);
                    if (bound != null) {
                        labelsBuilder.remove(LabelKeys.BIN);
                        Labels finalLabels = labelsBuilder.build();
                        MetricId metricId = new MetricId(name, finalLabels);
                        histogramMetrics.put(metricId, bound, value);
                    }
                } else {
                    Labels finalLabels = labelsBuilder.build();

                    if (metricKey.getType() == MetricType.COUNTER) {
                        writer.writeCounter(name, finalLabels, value);
                        serviceMetrics.registerCounter();

                    } else {
                        writer.writeGauge(name, finalLabels, value);
                        serviceMetrics.registerGauge();
                    }
                }
            }

            encodeSplitHist(serviceMetrics, histogramMetrics, writer);
        }
        billingReads(serviceMetrics.cloudId, serviceMetrics.folderId, now, stats);
        return out.toByteArray();
    }

    private void billingReads(String cloudId, String folderId, Instant now, EnumMapToInt<MetricType> stats) {
        long nowSec = TimeUnit.MILLISECONDS.toSeconds(now.toEpochMilli());
        for (var type : MetricType.values()) {
            int count = stats.get(type);
            if (count == 0) {
                continue;
            }

            var protoType = MetricTypeConverter.toNotNullProto(type);
            billingLog.readPoints(cloudId, folderId, protoType, count, nowSec, nowSec);
        }
    }

    private void encodeSplitHist(
            FolderServiceMetrics serviceMetrics,
            Table<MetricId, Double, Double> histogramMetrics,
            ExtendedPrometheusWriter writer)
    {
        for (var entry : histogramMetrics.rowMap().entrySet()) {
            MetricId metricId = entry.getKey();
            var sortedBuckets = entry.getValue()
                    .entrySet()
                    .stream()
                    .sorted(Map.Entry.comparingByKey())
                    .collect(Collectors.toList());

            double[] bounds = sortedBuckets.stream().mapToDouble(Map.Entry::getKey).toArray();
            double[] buckets = sortedBuckets.stream().mapToDouble(Map.Entry::getValue).toArray();
            writer.writeHistogram(metricId.getName(), metricId.getLabels(), bounds, buckets);
            serviceMetrics.registerHistogram();
        }
    }

    private static Double parseBinValue(String binValue) {
        if (binValue.equals("inf")) {
            return Histograms.INF_BOUND;
        }

        Double value = Doubles.tryParse(binValue);
        if (value != null && !Double.isFinite(value)) {
            return null;
        }
        return value;
    }

    private Selectors parseSelectors(String cloudId, String folderId, String service, String selectors) {
        try {
            return Selectors.parse(selectors)
                    .toBuilder()
                    .addOverride(Selector.exact(LabelKeys.PROJECT, cloudId))
                    .addOverride(Selector.exact(LabelKeys.CLUSTER, folderId))
                    .addOverride(Selector.exact(LabelKeys.SERVICE, service))
                    .build();
        } catch (Throwable e) {
            throw new BadRequestException(String.format("failed to parse selectors %s", selectors));
        }
    }
}
