package ru.yandex.solomon.gateway.api.cloud.v2;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.iam.ApiKeyAuthenticator;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.gateway.api.cloud.ext.ExternalMonitoringMetrics;
import ru.yandex.solomon.gateway.api.cloud.v1.CloudAuthorizer;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.spring.ConditionalOnBean;

/**
 * Simple Prometheus metrics controller.
 *
 * @author Oleg Baryshnikov
 */
@ApiIgnore
@RestController
@RequestMapping(path = "/monitoring/v2/prometheusMetrics", produces = MediaType.TEXT_PLAIN_VALUE)
@ConditionalOnBean(TGatewayCloudConfig.class)
@ParametersAreNonnullByDefault
public class PrometheusMetricsController {
    private final ApiKeyAuthenticator apiKeyAuthenticator;
    private final CloudAuthorizer authorizer;
    private final PrometheusMetricsClient client;
    private final SolomonConfHolder confHolder;

    @Autowired
    public PrometheusMetricsController(
        ApiKeyAuthenticator apiKeyAuthenticator,
        CloudAuthorizer authorizer,
        PrometheusMetricsClient client,
        SolomonConfHolder confHolder)
    {
        this.apiKeyAuthenticator = apiKeyAuthenticator;
        this.authorizer = authorizer;
        this.client = client;
        this.confHolder = confHolder;
    }

    @RequestMapping(method = RequestMethod.GET)
    public CompletableFuture<ResponseEntity<byte[]>> metrics(
        ServerHttpRequest request,
        @RequestParam("folderId") String folderId,
        @RequestParam("service") String service,
        @RequestParam(value = "selectors", defaultValue = "") String selectorsStr)
    {
        return apiKeyAuthenticator.getToken(request)
            .thenCompose(subject -> authorizer.authorizeAndResolveCloudId(subject, folderId, Permission.DATA_READ, cloudId -> {
                var future = metricsImpl(cloudId, folderId, service, selectorsStr, AuthSubject.getLogin(subject, subject.getUniqueId()));
                ExternalMonitoringMetrics.forFuture("/v2/prometheusMetrics", cloudId, folderId, future);
                return future;
            }));
    }

    private CompletableFuture<ResponseEntity<byte[]>> metricsImpl(
            String cloudId,
            String folderId,
            String service,
            String selectorsStr,
            String subjectId) {
        Instant now = Instant.now();
        Instant deadline = now.plus(1, ChronoUnit.MINUTES);

        var shard = confHolder.getConfOrThrow().findShardOrNull(new ShardKey(cloudId, folderId, service));
        if (shard != null) {
            return client.loadPrometheusMetrics(cloudId, folderId, service, selectorsStr, now, deadline, subjectId)
                    .thenApply(response -> new ResponseEntity<>(response, HttpStatus.OK));
        }
        return CompletableFuture.completedFuture(ResponseEntity.notFound().build());
    }
}
