package ru.yandex.solomon.gateway.api.cloud.v2.dto;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling.FillOption;
import ru.yandex.solomon.metrics.client.combined.DataLimits;

/**
 * @author Sergey Polovko
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DownsamplingDto {

    enum GridAggregation {
        MAX,
        MIN,
        SUM,
        AVG,
        LAST,
        COUNT,
        ;
    }

    enum GapFilling {
        NULL,
        NONE,
        PREVIOUS,
        ;
    }

    @JsonProperty
    private Long maxPoints;
    @JsonProperty
    private Long gridInterval;
    @JsonProperty
    private Boolean disabled;
    @JsonProperty
    private GridAggregation gridAggregation;
    @JsonProperty
    private GapFilling gapFilling;

    DownsamplingOptions toOptions() {
        var options = DownsamplingOptions.newBuilder()
                .setDownsamplingAggr(convertToEAggregation())
                .setDownsamplingFill(convertToDownsamplingFill());
        if (disabled != null && disabled) {
            options.setDownsamplingType(DownsamplingType.OFF);
            options.setGridMillis(0);
        } else if (maxPoints != null) {
            options.setDownsamplingType(DownsamplingType.BY_POINTS);
            options.setPoints(maxPoints.intValue());
        } else if (gridInterval != null) {
            options.setDownsamplingType(DownsamplingType.BY_INTERVAL);
            options.setGridMillis(gridInterval);
        } else {
            options.setDownsamplingType(DownsamplingType.BY_POINTS);
            options.setPoints(500);
        }
        return options.build();
    }

    private FillOption convertToDownsamplingFill() {
        if (gapFilling == null) {
            return FillOption.NULL;
        }
        switch (gapFilling) {
            case NULL: return FillOption.NULL;
            case NONE: return FillOption.NONE;
            case PREVIOUS: return FillOption.PREVIOUS;
            default:
                throw new RuntimeException("unknown gap filling: " + gapFilling);
        }
    }

    private Aggregation convertToEAggregation() {
        if (gridAggregation == null) {
            return Aggregation.DEFAULT_AGGREGATION;
        }
        switch (gridAggregation) {
            case MAX: return Aggregation.MAX;
            case MIN: return Aggregation.MIN;
            case SUM: return Aggregation.SUM;
            case AVG: return Aggregation.AVG;
            case LAST: return Aggregation.LAST;
            case COUNT: return Aggregation.COUNT;
            default:
                throw new RuntimeException("unknown grid aggregation: " + gridAggregation);
        }
    }

    public void validate() {
        int maxPoints = this.maxPoints != null ? this.maxPoints.intValue() : 0;
        long gridMillis = this.gridInterval == null ? 0 : this.gridInterval;

        if (maxPoints < 0) {
            throw new BadRequestException(maxPoints + " cannot be negative");
        }

        if (maxPoints > 0 && maxPoints < DataLimits.MIN_POINTS_COUNT) {
            throw new BadRequestException("too few points: " + maxPoints + " < " + DataLimits.MIN_POINTS_COUNT);
        }

        if (maxPoints > DataLimits.MAX_POINTS_COUNT) {
            throw new BadRequestException("too many points: " + maxPoints + " > " + DataLimits.MAX_POINTS_COUNT);
        }

        if (gridMillis < 0) {
            throw new BadRequestException(gridMillis + " cannot be negative");
        }
    }
}
