package ru.yandex.solomon.gateway.api.cloud.v2.dto;

import java.time.Instant;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.api.cloud.common.ProgramValidator;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Sergey Polovko
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ReadRequestDto {

    @JsonProperty
    private String query;
    @JsonProperty
    private Instant fromTime;
    @JsonProperty
    private Instant toTime;
    @JsonProperty
    private DownsamplingDto downsampling;

    public DataRequest toDataRequest(String cloudId, String folderId, RequestProducer producer, Instant deadline, String subjectId) {
        DataRequest.Builder builder = DataRequest.newBuilder();
        builder.setProjectId(cloudId);
        builder.setClusterName(folderId);
        builder.setProgram(Nullables.orEmpty(this.query));
        builder.setUseNewFormat(true);
        builder.setVersion(SelVersion.CURRENT);

        DownsamplingOptions downsamplingOptions = (this.downsampling != null)
            ? this.downsampling.toOptions()
            : DownsamplingOptions.newBuilder().build();
        builder.setDownsampling(downsamplingOptions);

        Interval interval = new Interval(this.fromTime, this.toTime);
        builder.setInterval(interval);

        builder.setProducer(producer);
        builder.setDeadline(deadline);
        builder.setSubjectId(subjectId);

        builder.setVersion(SelVersion.GROUP_LINES_RETURN_VECTOR_2);

        return builder.build();
    }

    public void validate() {
        if (this.downsampling != null) {
            this.downsampling.validate();
        }
        if (this.fromTime == null) {
            throw new BadRequestException("missed parameter: \"fromTime\"");
        }
        if (this.toTime == null) {
            throw new BadRequestException("missed parameter: \"toTime\"");
        }
        ValidationUtils.validateInterval(this.fromTime, this.toTime);
        if (StringUtils.isBlank(query)) {
            throw new BadRequestException("query cannot be empty");
        }

        ProgramValidator.validateMonitoringDataApiQuery(query);
    }
}
