package ru.yandex.solomon.gateway.api.cloud.v2.dto;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;

import static ru.yandex.solomon.model.protobuf.MetricType.DGAUGE;
import static ru.yandex.solomon.model.protobuf.MetricType.IGAUGE;

/**
 * @author Sergey Polovko
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class TimeseriesDto {

    @JsonProperty
    private final long[] timestamps;
    @JsonProperty
    private final double[] doubleValues;
    @JsonProperty
    private final long[] int64Values;

    private TimeseriesDto(long[] timestamps, double[] doubleValues, long[] int64Values) {
        this.timestamps = timestamps;
        this.doubleValues = doubleValues;
        this.int64Values = int64Values;
    }

    public static TimeseriesDto fromModel(MetricType metabaseType, ru.yandex.solomon.model.protobuf.MetricType dataType, AggrGraphDataIterable data) {
        return switch (metabaseType) {
            case DGAUGE, RATE -> toDouble(dataType, data);
            case IGAUGE, COUNTER -> toInt64(dataType, data);
            default -> throw new IllegalArgumentException("unknown metric type: " + metabaseType);
        };
    }

    private static TimeseriesDto toDouble(ru.yandex.solomon.model.protobuf.MetricType dataType, AggrGraphDataIterable data) {
        int count = data.getRecordCount();
        long[] timestamps = new long[count];
        double[] values = new double[count];

        if (count > 0) { // if count = 0 then dataType is METRIC_TYPE_UNSPECIFIED
            AggrGraphDataListIterator iterator = MetricTypeTransfers.of(dataType, DGAUGE, data.iterator());
            RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
            try {
                for (int i = 0; iterator.next(point); ++i) {
                    timestamps[i] = point.tsMillis;
                    values[i] = point.getValueDivided();
                }
            } finally {
                point.recycle();
            }
        }
        return new TimeseriesDto(timestamps, values, null);
    }

    private static TimeseriesDto toInt64(ru.yandex.solomon.model.protobuf.MetricType dataType, AggrGraphDataIterable data) {
        int count = data.getRecordCount();
        long[] timestamps = new long[count];
        long[] values = new long[count];

        if (count > 0) { // if count = 0 then dataType is METRIC_TYPE_UNSPECIFIED
            AggrGraphDataListIterator iterator = MetricTypeTransfers.of(dataType, IGAUGE, data.iterator());
            RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
            try {
                for (int i = 0; iterator.next(point); ++i) {
                    timestamps[i] = point.tsMillis;
                    values[i] = point.longValue;
                }
            } finally {
                point.recycle();
            }
        }
        return new TimeseriesDto(timestamps, null, values);
    }

    public long[] getTimestamps() {
        return timestamps;
    }

    public double[] getDoubleValues() {
        return doubleValues;
    }

    public long[] getInt64Values() {
        return int64Values;
    }
}
